import {createSelector} from 'reselect';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';

import {EDirection} from 'types/common/EDirection';
import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';

import {TOrderStepDescription} from 'reducers/trains/order/thunk/changeOrderStep';

import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';
import stepRestrictionsDataSelector from 'selectors/trains/order/stepRestrictionsDataSelector';
import {orderStepSelector} from 'selectors/trains/order/orderStepSelector';
import placesStepsSelector from 'selectors/trains/order/placesStepsSelector';
import orderInfoStateSelector from 'selectors/trains/order/orderInfoStateSelector';

import {getRestrictions} from 'projects/trains/lib/order/stepRestrictions/restrictions';

const availableStepsSelector = createSelector(
    [
        currentSegmentDirectionAndIndexSelector,
        stepRestrictionsDataSelector,
        orderStepSelector,
        orderInfoStateSelector,
        placesStepsSelector,
    ],
    (
        currentSegmentDirectionAndIndex,
        stepRestrictionsData,
        orderStep,
        orderInfoState,
        placesSteps,
    ): TOrderStepDescription[] => {
        const passengersStep = createStep(ORDER_STEP.PASSENGERS);
        const confirmStep = createStep(ORDER_STEP.CONFIRM);
        const paymentStep = createStep(ORDER_STEP.PAYMENT);

        switch (orderStep) {
            case ORDER_STEP.PLACES: {
                const {direction: currentDirection, index: currentIndex} =
                    currentSegmentDirectionAndIndex;
                const placesStepsBeforeAndSomeCurrent = placesSteps.filter(
                    ({direction, index}) => {
                        if (currentDirection === EDirection.FORWARD) {
                            return (
                                direction === EDirection.FORWARD &&
                                index <= currentIndex
                            );
                        }

                        return (
                            direction === EDirection.FORWARD ||
                            index <= currentIndex
                        );
                    },
                );

                const hasRestrictions =
                    getRestrictions(orderStep)(stepRestrictionsData).length;

                if (
                    placesStepsBeforeAndSomeCurrent.length ===
                    placesSteps.length
                ) {
                    return hasRestrictions
                        ? placesSteps
                        : [...placesSteps, passengersStep];
                }

                return hasRestrictions
                    ? placesStepsBeforeAndSomeCurrent
                    : [
                          ...placesStepsBeforeAndSomeCurrent,
                          placesSteps[placesStepsBeforeAndSomeCurrent.length],
                      ];
            }
            case ORDER_STEP.PASSENGERS: {
                return [...placesSteps, passengersStep];
            }
            case ORDER_STEP.CONFIRM: {
                return [...placesSteps, passengersStep, confirmStep];
            }
            case ORDER_STEP.PAYMENT: {
                return orderInfoState === EGenericOrderState.CONFIRMED
                    ? [paymentStep]
                    : [
                          ...placesSteps,
                          passengersStep,
                          confirmStep,
                          paymentStep,
                      ];
            }

            default: {
                return [];
            }
        }
    },
);

function createStep(
    step: ORDER_STEP.PASSENGERS | ORDER_STEP.CONFIRM | ORDER_STEP.PAYMENT,
): TOrderStepDescription {
    return {step};
}

export default availableStepsSelector;
