import flatMap from 'lodash/flatMap';
import times from 'lodash/times';

import {ITrainsSaveOfferServiceParams} from 'server/services/TrainsService/types/ITrainsSaveOfferService';
import {DIRECTIONS, EDirection} from 'types/common/EDirection';

import {StoreInterface} from 'reducers/storeTypes';

import directionTrainsCountSelector from 'selectors/trains/order/directionTrainsCountSelector';
import orderPriceDetailsSelector from 'selectors/trains/order/orderPriceDetailsSelector';
import passengersCountSelector from 'selectors/trains/order/passengersCountSelector';
import trainsSelector from 'selectors/trains/order/trainsSelector';
import passengersForOrderSelector from 'selectors/trains/order/passengersForOrderSelector';

import {prepareSaveOfferParams} from 'projects/trains/lib/api/utilities/prepareSaveOfferParams/prepareSaveOfferParams';

interface ISaveOfferParamsWithDirectionAndIndex {
    direction: EDirection;
    index: number;
    params: ITrainsSaveOfferServiceParams;
}

/**
 * Вычисляет параметры для создания офферов для каждого поезда в заказе
 *
 * без мемоизации тк не используется в рендер
 */
export default function saveOfferParamsSelector(
    state: StoreInterface,
): ISaveOfferParamsWithDirectionAndIndex[] {
    const trains = trainsSelector(state);
    const directionTrainsCount = directionTrainsCountSelector(state);
    const orderPriceDetails = orderPriceDetailsSelector(state);
    const passengers = passengersCountSelector(state);
    const allPassengersData = passengersForOrderSelector(state);

    return flatMap(DIRECTIONS, direction => {
        const trainCount = directionTrainsCount[direction];

        return times(trainCount, index => {
            const priceDetails = orderPriceDetails[direction][index];
            const passengersData = allPassengersData[direction][index];

            const train = trains[direction][index];
            const {
                coach,
                beddingOption,
                orderPlaces,
                requirements,
                segment,
                gender,
                trainDetails: trainDetailsInfo,
            } = train;

            const trainDetails = trainDetailsInfo?.trainDetails ?? null;

            const params = prepareSaveOfferParams({
                orderPlaces,
                beddingOption,
                requirements,
                segment,
                coach,
                priceDetails,
                gender,
                trainDetails,
                passengers,
                passengersData,
                direction,
                index,
            });

            return {
                direction,
                index,
                params,
            };
        });
    });
}
