import {createSelector} from 'reselect';
import flatMap from 'lodash/flatMap';

import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';
import {isNotNull} from 'types/utilities';

import orderPriceDetailsSelector from 'selectors/trains/order/orderPriceDetailsSelector';
import orderIsReservedSelector from 'selectors/trains/order/orderIsReservedSelector';
import insuranceIncludedSelector from 'selectors/trains/order/insuranceIncludedSelector';
import orderInfoSelector from 'selectors/trains/order/orderInfoSelector';

import IPrice from 'utilities/currency/PriceInterface';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import {getTotalInsuranceAmountByOrderInfo} from 'projects/trains/lib/order/insurance';
import sumPrice from 'utilities/price/sumPrice';
import {getTrainServices} from 'projects/trains/lib/order/getTrainServices';

export interface ITotalPriceSelector {
    /**
     * Общая итоговая цена
     */
    totalPrice: IPrice;
    /**
     * Общая итоговая цена для возвратных тарифов
     */
    refundableTotalPrice: IPrice;
    /**
     * Показать вопросик с детализацией итоговой цены
     */
    detailsIsVisible: boolean;
    /**
     * Цена сбора в итоговой цене
     */
    feePrice?: IPrice;
    /**
     * Цена билетов в итоговой цене
     */
    ticketsPrice?: IPrice;
    /**
     * Цена страховки в итоговой цене
     */
    insurancePrice?: IPrice;
}

const totalPriceSelector = createSelector(
    [
        orderPriceDetailsSelector,
        orderIsReservedSelector,
        insuranceIncludedSelector,
        orderInfoSelector,
    ],
    (
        orderPriceDetails,
        isReserved,
        insuranceIncluded,
        orderInfo,
    ): ITotalPriceSelector => {
        const insurancePrice = getTotalInsuranceAmountByOrderInfo(orderInfo);
        const {feePrice, ticketsPrice} = calculateDetails(orderInfo);

        return flatMap(
            Object.values(orderPriceDetails),
            directionPriceDetails => Object.values(directionPriceDetails),
        ).reduce<ITotalPriceSelector>(
            (acc, priceDetails) => {
                const {price, refundablePrice} = priceDetails;

                return {
                    ...acc,
                    totalPrice: sumPrice(acc.totalPrice, price),
                    refundableTotalPrice: sumPrice(
                        acc.refundableTotalPrice,
                        refundablePrice,
                    ),
                };
            },
            {
                totalPrice: {value: 0, currency: CurrencyType.RUB},
                refundableTotalPrice: {value: 0, currency: CurrencyType.RUB},
                detailsIsVisible: isReserved,
                insurancePrice:
                    insuranceIncluded && insurancePrice
                        ? insurancePrice
                        : undefined,
                feePrice,
                ticketsPrice,
            },
        );
    },
);

export default totalPriceSelector;

interface IDetails {
    feePrice: IPrice | undefined;
    ticketsPrice: IPrice | undefined;
}

/**
 * Возвращает суммарные значения частичных payment'ов, если они существуют
 * для всех поездов в заказе
 */
function calculateDetails(orderInfo: IGenericOrderInfo | null): IDetails {
    const trainServices = getTrainServices(orderInfo);

    const tickets = flatMap(trainServices, service =>
        service.trainInfo.passengers.map(p => p.ticket),
    ).filter(isNotNull);

    if (!tickets.length) {
        return {
            feePrice: undefined,
            ticketsPrice: undefined,
        };
    }

    return {
        feePrice: sumPrice(...tickets.map(ticket => ticket.payment.fee)),
        ticketsPrice: sumPrice(...tickets.map(ticket => ticket.payment.amount)),
    };
}
