import {createSelector} from 'reselect';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';

import {
    IStation,
    IThread,
    TTrainsOrderSelectorSegment,
} from 'projects/trains/lib/segments/types';
import {ITrainsTariffs} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import {orderStepSelector} from 'selectors/trains/order/orderStepSelector';
import trainsSelector from 'selectors/trains/order/trainsSelector';

import getTrainNumber from 'projects/trains/lib/order/getTrainNumber';
import getSegmentTimes from 'projects/trains/lib/segments/getSegmentTimes';
import mapTrainsState, {
    TMapTrainsState,
} from 'projects/trains/lib/complexOrder/mapTrainsState';

export type TOrderTrainsSegments =
    TMapTrainsState<TTrainsOrderSelectorSegment | null>;

const orderTrainsSegmentsSelector = createSelector(
    [trainsSelector, orderStepSelector],
    (trains, orderStep: ORDER_STEP): TOrderTrainsSegments => {
        return mapTrainsState(trains, train => {
            const {segment, coach, trainDetails: trainDetailsInfo} = train;
            const trainDetails = trainDetailsInfo?.trainDetails;

            if (segment === null) {
                return null;
            }

            const patchedSegmentProps: {
                arrival: string;
                number: string;
                tariffs:
                    | undefined
                    | {
                          electronicTicket: boolean;
                      }
                    | ITrainsTariffs;
                thread: IThread | null;
                stationFrom: IStation;
                stationTo: IStation;
                originalArrival?: string;
                isThroughCoachArrival?: boolean;
            } = {
                arrival: segment.arrival,
                number: segment.number,
                tariffs: 'tariffs' in segment ? segment.tariffs : undefined,
                thread: segment.thread,
                stationFrom: segment.stationFrom,
                stationTo: segment.stationTo,
            };

            if (
                orderStep !== ORDER_STEP.PLACES &&
                coach &&
                coach.throughArrival
            ) {
                patchedSegmentProps.isThroughCoachArrival = true;
                patchedSegmentProps.arrival = coach.throughArrival;
                patchedSegmentProps.originalArrival = segment.arrival;
            }

            if (trainDetails) {
                patchedSegmentProps.number = getTrainNumber(
                    segment,
                    trainDetails,
                );

                patchedSegmentProps.tariffs = ('tariffs' in segment &&
                    segment.tariffs) || {
                    electronicTicket: Boolean(trainDetails.electronicTicket),
                };

                patchedSegmentProps.stationFrom = {
                    ...segment.stationFrom,
                    settlement: trainDetails.stationFrom.settlement,
                };

                patchedSegmentProps.stationTo = {
                    ...segment.stationTo,
                    settlement: trainDetails.stationTo.settlement,
                };

                // Если есть название фирменного поезда от партнера, то используем его
                if (
                    patchedSegmentProps.thread &&
                    !patchedSegmentProps.thread.deluxeTrain &&
                    trainDetails.rawTrainName
                ) {
                    patchedSegmentProps.thread.deluxeTrain = {
                        shortTitle: trainDetails.rawTrainName,
                        title: trainDetails.rawTrainName,
                        isDeluxe: false,
                        isHighSpeed: false,
                        id: 0,
                    };
                }
            }

            return {
                ...segment,
                ...patchedSegmentProps,
                ...getSegmentTimes({
                    ...segment,
                    arrival: patchedSegmentProps.arrival || segment.arrival,
                }),
            };
        });
    },
);

export default orderTrainsSegmentsSelector;
