import {createSelector} from 'reselect';

import {EDirection} from 'types/common/EDirection';
import {TSelector} from 'src/redux/types/TSelector';
import {ECalendarType} from 'components/Calendar/types';
import {
    TCalendarPrices,
    ICalendarPricesByType,
} from 'types/common/calendarPrice/ICalendarPrice';
import {ITrainsPriceCalendarDateWithPrice} from 'types/trains/search/priceCalendar/ITrainsPriceCalendar';

import {ITrainsPriceCalendarReducer} from 'reducers/trains/priceCalendar/reducer';

import getIsNewCalendarEnabled from 'selectors/common/getIsNewCalendarEnabled';

const getPriceCalendar: TSelector<ITrainsPriceCalendarReducer> = state =>
    state.trains.priceCalendar;

const calculateCalendarPrices = ({
    priceDates,
}: {
    priceDates: ITrainsPriceCalendarDateWithPrice[];
}): TCalendarPrices | undefined => {
    if (!priceDates?.length) {
        return;
    }

    return priceDates.reduce<TCalendarPrices>(
        (calendarPriceMap, dateWithPrice) => {
            const {date, price, emptyPriceReason} = dateWithPrice;

            if (price?.value) {
                calendarPriceMap[date] = {
                    roughly: false,
                    ...price,
                };
            }

            if (emptyPriceReason) {
                calendarPriceMap[date] = {
                    emptyPriceReason,
                };
            }

            return calendarPriceMap;
        },
        {},
    );
};

export const priceCalendarSelector = createSelector(
    getPriceCalendar,
    getIsNewCalendarEnabled,
    (
        priceCalendar: ITrainsPriceCalendarReducer,
        isNewCalendarEnabled: boolean,
    ): ICalendarPricesByType | undefined => {
        const forwardPriceDates = priceCalendar[EDirection.FORWARD]?.dates;
        const backwardPriceDates = priceCalendar[EDirection.BACKWARD]?.dates;

        if (
            isNewCalendarEnabled ||
            (!forwardPriceDates?.length && !backwardPriceDates?.length)
        ) {
            return;
        }

        return {
            [ECalendarType.StartDate]: calculateCalendarPrices({
                priceDates: forwardPriceDates,
            }),
            [ECalendarType.EndDate]: calculateCalendarPrices({
                priceDates: backwardPriceDates,
            }),
        };
    },
);
