import {createSelector} from 'reselect';

import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {
    ITrainsSearchFormPointField,
    TTrainsSearchFormDateField,
} from 'projects/trains/components/SearchForm/types';

import {StoreInterface} from 'reducers/storeTypes';

import deviceTypeSelector from 'selectors/common/deviceTypeSelector';
import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import {fromFieldSelector} from 'selectors/trains/searchForm/fromFieldSelector';
import {toFieldSelector} from 'selectors/trains/searchForm/toFieldSelector';

import isAllDaysSearch from 'projects/trains/lib/search/isAllDaysSearch';
import {formatDateRange} from 'utilities/dateUtils';
import getSearchInformationHumanStartDate from 'utilities/searchForm/getSearchInformationHumanStartDate';

import * as i18nBlock from 'i18n/trains-search-information';

const calculateSearchInformationWhen = (
    startDate?: TTrainsSearchFormDateField,
    endDate?: TTrainsSearchFormDateField,
): string => {
    if (startDate && endDate) {
        return formatDateRange(startDate, endDate);
    }

    if (startDate) {
        return getSearchInformationHumanStartDate(startDate);
    }

    return i18nBlock.periodDashPlaceholder();
};

export default createSelector(
    deviceTypeSelector,
    trainsContextSelector,
    fromFieldSelector,
    toFieldSelector,
    (state: StoreInterface) =>
        state.trains.searchForm[ESearchFormFieldName.START_DATE],
    (state: StoreInterface) =>
        state.trains.searchForm[ESearchFormFieldName.END_DATE],
    (
        deviceType,
        context,
        from: ITrainsSearchFormPointField,
        to: ITrainsSearchFormPointField,
        startDate: TTrainsSearchFormDateField,
        endDate: TTrainsSearchFormDateField,
    ) => ({
        from: (from && from.inputValue) || i18nBlock.fromDashPlaceholder(),
        to: (to && to.inputValue) || i18nBlock.toDashPlaceholder(),
        when: calculateSearchInformationWhen(startDate, endDate),
        withHeading:
            isFilledTrainsSearchContext(context) &&
            deviceType.isMobile &&
            !isAllDaysSearch(context),
    }),
);
