import {
    exponentialDelay,
    IAxiosRetryConfig,
    isNetworkError,
    isRetryableError,
} from 'axios-retry';

import IGenericOrderService from 'server/services/GenericOrderService/types/IGenericOrderService';
import IGenericOrderAddServiceApiParams from 'server/api/GenericOrderApi/types/addService/IGenericOrderAddServiceApiParams';
import IGenericOrderAddServiceApiResponse from 'server/api/GenericOrderApi/types/addService/IGenericOrderAddServiceApiResponse';
import IGenericOrderCalculateRefundAmountApiParams from 'server/api/GenericOrderApi/types/calculateRefundAmount/IGenericOrderCalculateRefundAmountApiParams';
import IGenericOrderCalculateRefundAmountApiResponse from 'server/api/GenericOrderApi/types/calculateRefundAmount/IGenericOrderCalculateRefundAmountApiResponse';
import IGenericOrderCancelOrderApiParams from 'server/api/GenericOrderApi/types/cancelOrder/IGenericOrderCancelOrderApiParams';
import IGenericOrderCreateOrderServiceParams from 'server/services/GenericOrderService/types/createOrder/IGenericOrderCreateOrderServiceParams';
import IGenericOrderCreateOrderApiResponse from 'server/api/GenericOrderApi/types/createOrder/IGenericOrderCreateOrderApiResponse';
import IGenericOrderEstimateDiscountApiParams from 'server/api/GenericOrderApi/types/estimateDiscount/IGenericOrderEstimateDiscountApiParams';
import IGenericOrderGetOrderApiParams from 'server/api/GenericOrderApi/types/getOrder/IGenericOrderGetOrderApiParams';
import IGenericOrderGetOrderStateApiResponse from 'server/api/GenericOrderApi/types/getOrderState/IGenericOrderGetOrderStateApiResponse';
import IGenericOrderEstimateDiscountApiResponse from 'server/api/GenericOrderApi/types/estimateDiscount/IGenericOrderEstimateDiscountApiResponse';
import IGenericOrderGetOrderStateApiParams from 'server/api/GenericOrderApi/types/getOrderState/IGenericOrderGetOrderStateApiParams';
import IGenericOrderStartPaymentApiParams from 'server/api/GenericOrderApi/types/startPayment/IGenericOrderStartPaymentApiParams';
import IGenericOrderStartRefundApiParams from 'server/api/GenericOrderApi/types/startRefund/IGenericOrderStartRefundApiParams';
import IGenericOrderGetOrderApiResponse from 'server/api/GenericOrderApi/types/getOrder/IGenericOrderGetOrderApiResponse';
import IGenericOrderDownloadBlankApiParams from 'server/api/GenericOrderApi/types/downloadBlank/IGenericOrderDownloadBlankApiParams';

import {BrowserHttpClient} from 'utilities/browserHttpClient/BrowserHttpClient';

const createRetryConfig = (): IAxiosRetryConfig => ({
    retryCondition: (error): boolean =>
        isNetworkError(error) || isRetryableError(error),
    retryDelay: exponentialDelay,
    retries: 3,
});

export class GenericOrderBrowserProvider
    extends BrowserHttpClient
    implements IGenericOrderService
{
    constructor() {
        super({baseURL: '/api/genericOrder'});
    }

    addService = (
        params: IGenericOrderAddServiceApiParams,
    ): Promise<IGenericOrderAddServiceApiResponse> => {
        return this.post('addService', params, {
            axiosRetry: createRetryConfig(),
        });
    };

    calculateRefundAmount = (
        params: IGenericOrderCalculateRefundAmountApiParams,
    ): Promise<IGenericOrderCalculateRefundAmountApiResponse> => {
        return this.post('calculateRefundAmount', params, {
            axiosRetry: createRetryConfig(),
        });
    };

    cancelOrder = (
        params: IGenericOrderCancelOrderApiParams,
    ): Promise<void> => {
        return this.post('cancelOrder', params, {
            axiosRetry: createRetryConfig(),
        });
    };

    createOrder = (
        params: IGenericOrderCreateOrderServiceParams,
    ): Promise<IGenericOrderCreateOrderApiResponse> => {
        return this.post('createOrder', params, {
            axiosRetry: createRetryConfig(),
        });
    };

    estimateDiscount = (
        params: IGenericOrderEstimateDiscountApiParams,
    ): Promise<IGenericOrderEstimateDiscountApiResponse> => {
        return this.post('estimateDiscount', params, {
            axiosRetry: createRetryConfig(),
        });
    };

    getOrder = (
        params: IGenericOrderGetOrderApiParams,
        abortParams?: {signal: AbortSignal},
    ): Promise<IGenericOrderGetOrderApiResponse> => {
        return this.get('getOrder', {
            params,
            axiosRetry: createRetryConfig(),
            ...abortParams,
        });
    };

    getOrderState = (
        params: IGenericOrderGetOrderStateApiParams,
        abortParams?: {signal: AbortSignal},
    ): Promise<IGenericOrderGetOrderStateApiResponse> => {
        return this.get('getOrderState', {
            params,
            axiosRetry: createRetryConfig(),
            ...abortParams,
        });
    };

    startPayment = (
        params: IGenericOrderStartPaymentApiParams,
    ): Promise<void> => {
        return this.post('startPayment', params, {
            axiosRetry: createRetryConfig(),
        });
    };

    startRefund = (
        params: IGenericOrderStartRefundApiParams,
    ): Promise<void> => {
        return this.post('startRefund', params, {
            axiosRetry: createRetryConfig(),
        });
    };

    downloadBlank = (
        params: IGenericOrderDownloadBlankApiParams,
    ): Promise<NodeJS.ReadableStream> => {
        return this.get('downloadBlank', {params});
    };
}

export const genericOrderBrowserProvider = new GenericOrderBrowserProvider();
