import React, {useMemo} from 'react';

import resolveVariables from './utilities/resolveVariables';
import colorsVars from './utilities/colorsVars';
import resolveRgba from './utilities/resolveRgba';
import getUsagesUrl from './utilities/getUsagesUrl';
import {useBoolean} from 'utilities/hooks/useBoolean';

import Box from 'components/Box/Box';
import Input from 'components/Input/Input';
import Heading from 'components/Heading/Heading';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import SearchIcon from 'icons/12/Search';
import CheckboxBox from 'components/Checkbox/Checkbox';

import cx from './ColorsShowcase.scss';

const ColorsShowcase: React.FC = () => {
    const [filter, setFilter] = React.useState('');

    const {value: isDarkTheme, setValue: setIsDarkTheme} = useBoolean(false);

    const colors = useMemo(() => {
        const current = isDarkTheme ? colorsVars.dark : colorsVars.light;

        return Object.fromEntries(
            Object.entries(current).map(([key, value]) => {
                return [
                    key.replace(/^(light-)|(dark-)/, ''),
                    value.replace(/^(\$light-)|(\$dark-)/, '$'),
                ];
            }),
        );
    }, [isDarkTheme]);

    const filteredColors = React.useMemo(() => {
        return Object.entries(colors).filter(([name]) => {
            return name.toLowerCase().includes(filter.toLowerCase());
        });
    }, [colors, filter]);

    return (
        <Box className={cx('root')} inset="4">
            <div className={cx('container')}>
                <Box below="6">
                    <Flex
                        inline
                        justifyContent="space-between"
                        alignItems="baseline"
                    >
                        <Heading level={1}>Список цветов</Heading>
                        <CheckboxBox
                            className={cx('darkThemeCheckbox')}
                            label="Темная тема"
                            checked={isDarkTheme}
                            onChange={(e): void =>
                                setIsDarkTheme(e.target.checked)
                            }
                        />
                    </Flex>
                    <Flex inline above={2} between={4}>
                        <Link
                            url="https://www.figma.com/file/I8EBpOVkBHKTHVrDyWuaaIhI/Travel.Styles?node-id=1036%3A168"
                            target="_blank"
                        >
                            Figma
                        </Link>
                        <Link
                            url="https://a.yandex-team.ru/arc_vcs/travel/frontend/portal/src/styles/colors.scss"
                            target="_blank"
                        >
                            Arcanum
                        </Link>
                        <Link
                            url="https://a.yandex-team.ru/arcadia/travel/frontend/portal/docs/design.md#cveta"
                            target="_blank"
                        >
                            Docs
                        </Link>
                    </Flex>
                </Box>
                <Box className={cx('filter')} below="4">
                    <Input
                        value={filter}
                        placeholder="filter colors"
                        onChange={(evt): void => setFilter(evt.target.value)}
                    />
                </Box>
                <Flex flexDirection="column" between={2}>
                    {filteredColors.map(([name, value]) => {
                        const resolvedValue = resolveVariables(value, colors);

                        return (
                            <Flex
                                key={name}
                                className={cx('colorItem')}
                                inline
                                between={2}
                                alignItems="flex-start"
                                nowrap
                                y={1}
                            >
                                <div
                                    className={cx('colorBox')}
                                    style={{
                                        background: resolveRgba(resolvedValue),
                                    }}
                                />
                                <Flex flexDirection="column" between={1}>
                                    <Text>{name}</Text>
                                    <Text size="s" color="secondary">
                                        <span title={resolvedValue}>
                                            {value}
                                        </span>
                                    </Text>
                                </Flex>
                                <Link
                                    className={cx('usagesLink')}
                                    url={getUsagesUrl(name)}
                                    target="_blank"
                                >
                                    <TextWithIcon
                                        size="s"
                                        text="Найти использования"
                                        iconLeft={SearchIcon}
                                    />
                                </Link>
                            </Flex>
                        );
                    })}
                </Flex>
            </div>
        </Box>
    );
};

export default ColorsShowcase;
