import {IResetFilterInfo} from 'server/api/HotelsSearchAPI/types/IResetFilterInfo';

import IPrice from 'utilities/currency/PriceInterface';

/* Active Filters Params */
export interface IFilterParams {
    filterAtoms?: TFilterAtomType[];
    filterPriceFrom?: number;
    filterPriceTo?: number;
    // Фильтрация по текущему geo id поиска. True - фильтрует, False - нет.
    // Если флаг не указан, то бэк выставляет дефолтное значение true
    onlyCurrentGeoId?: boolean;
}

/* Atom */
export type TFilterAtomType = string;

/* Base Filter */
export interface IBasicFilter {
    name: string;
    id: string;
    effect?: string;
    enabled: boolean;
    hint: string;
    atoms: TFilterAtomType[];
}

export enum EQuickControlType {
    QUICK_FILTER = 'quick-filter',
    QUICK_FILTER_PRICE = 'quick-price-filter',
    QUICK_SORT = 'quick-sort',
}
export interface IQuickControl {
    // Уникальный идентификатор, нужен для оптимизации верстки
    id: string;
    type: EQuickControlType;
    // Эффект, как отображать этот фильтр.
    // Примеры: подкраска рейтинга, спец-значок для "только спецпредложения".
    // Примеры: для быстрого фильтра -- какой-нибудь спец-цвет или другая штука.
    effect?: EQuickFilterEffect;
    // Если enabled == true, то можно на контрол нажимать.
    // Если enabled == false, то нажимать нельзя (будет пустая выдача).
    enabled: boolean;
}

// Быстрый фильтр
export type TQuickFilter = IQuickControl & {
    // Название.
    name: string;
    // всплывающая подсказка. Если отсутствует - то её нет
    hint?: string;

    // Для определения факта нажатости быстрофильтра:
    // если в текущем стейте фильтра все атомы из atomsOn есть, и нет ни одного из atomsOff - то быстрофильтр нажат
    // При нажатии на быстрофильтр: все атомы из atomsOff - удаляем из стейта, Все атомы из atomsOn - добавляем в стейт.
    // При отжатии быстрофильтра: Все атомы из atomsOn удаляем из стейта. atomsOff в стейте быть не могло, иначе фильтр бы не был нажат
    atomsOn: TFilterAtomType[];
    atomsOff: TFilterAtomType[];

    type: EQuickControlType.QUICK_FILTER;
};

export type TQuickPriceFilter = IQuickControl & {
    // Название
    name: string;
    type: EQuickControlType.QUICK_FILTER_PRICE;
};

export type TQuickSort = IQuickControl & {
    type: EQuickControlType.QUICK_SORT;
};

export type TQuickControlTypes = TQuickPriceFilter | TQuickSort | TQuickFilter;

export enum EQuickFilterEffect {
    YANDEX_PLUS = 'yandex-plus',
    MIR_CASHBACK = 'mir-cashback',
}

export enum EBasicFilterGroupId {
    PANSION = 'pansion',
    STARS = 'stars',
    PLACES_NEARBY = 'places-nearby',
    FACILITIES = 'facilities',
    RATINGS = 'ratings',
    DISTANCE_TO_CITY_CENTER = 'distance-to-city-center',
    CATEGORY = 'category',
    OPTIONS_MIR = 'options-mir',
    PARTNERS = 'partners',
    HOTEL_CHAINS = 'hotel-chains',
}

/* Filter Group */
export enum EBasicFilterGroupType {
    SINGLE = 'SINGLE',
    MULTI = 'MULTI',
}

export interface IBasicFilterGroup {
    name: string;
    id: EBasicFilterGroupId | string;
    type: EBasicFilterGroupType;
    items: IBasicFilter[];
}

export enum EDetailedFiltersBatchItemType {
    /**
     * фильтр по цене, надо взять его из IFilterInfo
     */
    PRICE = 'PRICE',
    /**
     * обычная группа фильтров (передаётся в detailedFilters)
     */
    GROUP = 'GROUP',
}

export interface IDetailedFilterItem {
    /**
     * Что должно быть в этом элементе
     */
    type: EDetailedFiltersBatchItemType;

    // Заполняется, если type == 'GROUP'
    detailedFilters?: IBasicFilterGroup;
}

export interface IDetailedFiltersBatch {
    items: IDetailedFilterItem[];
}

export interface IGeoIdFilter {
    selected: boolean;
    name: string;
}

/* Price Filter */
export interface IRangePriceFilter extends IPriceFilter {
    minValue?: number;
    maxValue?: number;
}

export interface IRangePriceFilterEstimateValues {
    minPriceEstimate: IPrice['value'];
    maxPriceEstimate: IPrice['value'];
}

export interface IPriceFilter extends IRangePriceFilterEstimateValues {
    currency: IPrice['currency'];
    histogramBounds?: number[];
    histogramCounts?: number[];
}

/* Change Filter Payload */
export interface IChangeFiltersTargetType {
    targetFiltersType: 'CURRENT' | 'ALL';
}

export interface IChangeFilterGroupPayload {
    atomsOn?: TFilterAtomType[];
    atomsOff?: TFilterAtomType[];
}

export interface IChangeFilterGroupMeta {
    filterId: string;
    itemId: string;
}

export interface IChangeFilterGroupPayloadWithTarget
    extends IChangeFilterGroupPayload,
        IChangeFiltersTargetType {}

export interface IChangePriceFilterPayload {
    minValue: number;
    maxValue: number;
}

export interface IChangePriceFilterPayloadWithTarget
    extends IChangePriceFilterPayload,
        IChangeFiltersTargetType {}

export interface IChangeTextFilterPayloadWithTarget
    extends IChangeFiltersTargetType {}

export interface ICurrentGeoIdFilterPayloadWithTarget
    extends IChangeFiltersTargetType {
    selected: boolean;
}

/* Filters Info */

export type TActiveFilterAtomsType = Record<TFilterAtomType, boolean>;

export type TCollapsedFilters = PartialRecord<string, true>;

export interface IFiltersInfo {
    params: IFilterParams;
    quickFilters: TQuickFilter[];
    detailedFilters?: IDetailedFilterItem[];
    detailedFiltersBatches: IDetailedFiltersBatch[];
    priceFilter: IPriceFilter;
    geoIdFilter: IGeoIdFilter | null;
    resetFilterInfo: IResetFilterInfo | null; // Информация для сброса фильтров. Если null, то блок сброса фильтров не рисуем
}

export interface ISearchControlsInfo {
    quickControls: TQuickControlTypes[];
}

export interface IFullFilters {
    foundHotelCount?: number;
    totalHotelCount?: number;
    quickFilters?: TQuickFilter[];
    quickControls?: TQuickControlTypes[];
    detailedFilters?: IDetailedFilterItem[];
    detailedFiltersBatches?: IDetailedFiltersBatch[];
    priceFilter?: IRangePriceFilter;
    activeFilterAtoms: TActiveFilterAtomsType;
    geoIdFilter?: IGeoIdFilter | null;
}

export interface IFullFiltersInfo {
    needSyncPermanentFiltersWithServer: boolean;
    needSyncCurrentFiltersWithServer: boolean;
    permanentFilters: IFullFilters;
    currentFilters: IFullFilters;
    totalActiveFilters: number;
    isLeftFiltersEnabled: boolean;
    collapsedFilters: TCollapsedFilters;
}
