import {Action} from 'redux';

// Remove types from T that are assignable to U
export type Diff<T, U> = T extends U ? never : T;

// Remove types from T that are not assignable to U
export type Filter<T, U> = T extends U ? T : never;

/**
 * Выводит типы экшенов.
 *
 * @example
 * import * as actions from './actions';
 * type ActionTypes = DeriveActionsTypes<typeof actions>;
 *
 * @example
 * const incrementActionCreator = () => ({type: INC})
 * const decrementActionCreator = () => ({type: DEC})
 *
 * type MyActionTypes = DeriveActionsTypes<{
 *     // Имя свойства не имеет значения и игнорируется
 *     inc: incrementActionCreator,
 *     dec: decrementActionCreator
 * }>
 * // MyActionTypes = {type: 'INC'} | {type: 'DEC'}
 */
export type DeriveActionsTypes<T extends Record<string, unknown>> = Filter<
    {
        [P in keyof T]: T[P] extends (...args: any[]) => any
            ? ReturnType<T[P]>
            : never;
    }[keyof T],
    Action
>;

type TNotNullable<T> = T extends null | undefined ? never : T;

/**
 * Вывод типа промиса возвращаемого функцией
 * Позволяет получить возвращаемый тип api запроса
 */
export type TPromiseReturnType<T extends (...args: any) => Promise<unknown>> =
    Awaited<ReturnType<T>>;

export function isNotNull<T>(value: T): value is Exclude<T, null> {
    return value !== null;
}

export function isNotUndefined<T>(value: T): value is Exclude<T, undefined> {
    return value !== undefined;
}

export function isNotNullable<T>(value?: T): value is TNotNullable<T> {
    return Boolean(value);
}

export function isActionLike(action: any): action is Action {
    return action && 'type' in action;
}

export function isUnknownObject(
    x: unknown,
): x is {[key in PropertyKey]: unknown} {
    return x !== null && typeof x === 'object';
}
