import {Reducer, Action} from 'typesafe-actions';

type TReducerInfo<State, Actions extends Action> = [
    State,
    Reducer<State, Actions>,
];

type TCallback<State> = (state: State, action: Action) => State;

/**
 * Возвращает коллбэк для handleActions, который вернет текущий state, если
 * стэйт для переданных редьюсеров не изменился, или выполнит функцию,
 * переданную в callback-параметре
 */
export default function callIfReducersChanged<State>(
    reducersInfo: TReducerInfo<any, Action<any>>[],
    callback: TCallback<State>,
): TCallback<State> {
    const reducersState = reducersInfo.map(([initialState]) => initialState);

    return (state, action): State => {
        const newReducersState = reducersInfo.map(([_, reducer], index) =>
            reducer(reducersState[index], action),
        );

        let needUpdate = false;

        reducersState.forEach((prevState, index) => {
            const newState = newReducersState[index];

            if (newState !== prevState) {
                needUpdate = true;
            }

            reducersState[index] = newState;
        });

        if (needUpdate) {
            return callback(state, action);
        }

        return state;
    };
}
