import {
    EGeoLocationStatus,
    TBrowserGeolocationData,
} from 'types/common/browserGeolocation';

const REJECT_TIMEOUT = 10000;

const composeLocationData: (
    coords: GeolocationCoordinates | null,
    status: EGeoLocationStatus,
) => TBrowserGeolocationData = (coords, status) => {
    return {
        sortOrigin: coords
            ? `${coords.longitude},${coords.latitude}`
            : undefined,
        geoLocationStatus: status,
    };
};

const browserGeolocation: {
    coords: GeolocationCoordinates | null;
    geoLocationStatus: EGeoLocationStatus;
    getGeolocation: () => Promise<TBrowserGeolocationData>;
} = {
    coords: null,
    geoLocationStatus: EGeoLocationStatus.UNKNOWN,
    async getGeolocation() {
        if (
            !this.coords ||
            this.geoLocationStatus !== EGeoLocationStatus.AVAILABLE
        ) {
            try {
                const position = await new Promise<GeolocationPosition>(
                    (resolve, reject) => {
                        setTimeout(reject, REJECT_TIMEOUT);
                        navigator.geolocation.getCurrentPosition(
                            data => resolve(data),
                            err => reject(err),
                            {
                                enableHighAccuracy: true,
                            },
                        );
                    },
                );

                this.coords = position.coords;
                this.geoLocationStatus = EGeoLocationStatus.AVAILABLE;
            } catch (err) {
                this.geoLocationStatus = EGeoLocationStatus.FORBIDDEN;
            }
        }

        return composeLocationData(this.coords, this.geoLocationStatus);
    },
};

export default browserGeolocation;
