import axios, {AxiosInstance, AxiosRequestConfig} from 'axios';
import axiosRetry from 'axios-retry';

import {BASE_AXIOS_HEADERS, X_RETPATH_Y} from 'constants/axios';
import {JAEGER_TRACER_OPTIONS} from 'server/constants/common';
import {ETestHeaders} from 'server/constants/headers';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';

import {
    IHttpClient,
    IRequestConfig,
} from 'server/utilities/HttpClient/IHttpClient';
import {csrf} from 'utilities/csrfToken/createGlobalTokenGetter';
import appData from 'utilities/appData/appData';
import {getStoredSrcParams} from 'utilities/srcParams/srcParams';
import {reachGoal} from 'utilities/metrika';
import pageManager from 'utilities/pageManager/pageManager';

const {TRACER_DEBUG_TAG_NAME} = JAEGER_TRACER_OPTIONS;

interface IBrowserHttpClient {
    baseURL: string;
    timeout?: number;
    httpClientConfig?: AxiosRequestConfig;
}

type TApiMethods = 'GET' | 'DELETE' | 'POST' | 'PUT' | 'PATCH';

const CSRF_METHODS = ['POST', 'PUT', 'DELETE', 'PATCH'] as TApiMethods[];

export class BrowserHttpClient {
    protected httpClient: IHttpClient;

    protected timeout = 25000;

    private readonly axiosInstance: AxiosInstance;

    protected constructor({
        baseURL,
        httpClientConfig = {},
        timeout,
    }: IBrowserHttpClient) {
        this.timeout = timeout || this.timeout;

        const axiosInstance = axios.create({
            baseURL,
            ...httpClientConfig,
        });

        this.axiosInstance = axiosInstance;
        this.httpClient = axiosInstance;
    }

    protected interceptRequest(_request: IRequestConfig): void | Promise<void> {
        return;
    }

    protected get<T>(path: string, options?: IRequestConfig): Promise<T> {
        return this.fetch<T>('GET', path, null, options);
    }

    protected delete<T>(path: string, options?: IRequestConfig): Promise<T> {
        return this.fetch<T>('DELETE', path, null, options);
    }

    protected post<T>(
        path: string,
        body?: unknown,
        options?: IRequestConfig,
    ): Promise<T> {
        return this.fetch<T>('POST', path, body, options);
    }

    protected put<T>(
        path: string,
        body?: unknown,
        options?: IRequestConfig,
    ): Promise<T> {
        return this.fetch<T>('PUT', path, body, options);
    }

    protected patch<T>(
        path: string,
        body?: unknown,
        options?: IRequestConfig,
    ): Promise<T> {
        return this.fetch<T>('PATCH', path, body, options);
    }

    private async fetch<T>(
        method: TApiMethods,
        path: string,
        body?: unknown,
        options?: IRequestConfig,
    ): Promise<T> {
        const requestConfig = this.createRequestConfig(
            path,
            method,
            body,
            options,
        );

        await this.interceptRequest(requestConfig);

        axiosRetry(
            this.axiosInstance,
            requestConfig.axiosRetry || {retries: 0},
        );

        const response = await this.httpClient.request(requestConfig);

        await this.tryRedirectToCaptcha(response.data);

        return response.data;
    }

    private tryRedirectToCaptcha(responseData: any): Promise<void> {
        if (!responseData) {
            return Promise.resolve();
        }

        const {type, captcha} = responseData;

        if (type !== 'captcha' || !captcha) {
            return Promise.resolve();
        }

        const captchaPage = captcha['captcha-page'];

        if (!captchaPage) {
            return Promise.resolve();
        }

        reachGoal(ECommonGoal.CAPTCHA_REDIRECT);
        window.location.href = captchaPage;

        return new Promise(resolve => {
            // Чтобы запрос не падал и не появлялась страница ошибки перед тем как произойдет редирект
            setTimeout(resolve, 1000);
        });
    }

    private createRequestConfig(
        url: string,
        method: TApiMethods,
        body: unknown,
        options?: IRequestConfig,
    ): IRequestConfig {
        const {headers = {}, params = {}, ...requestOptions} = options || {};
        const jaegerDebugId = appData.getByKey('jaegerDebugId');
        const jaegerDebugHeader = jaegerDebugId
            ? {[TRACER_DEBUG_TAG_NAME]: jaegerDebugId}
            : {};

        return {
            url,
            method,
            timeout: this.timeout,
            data: body,
            headers: {
                ...BASE_AXIOS_HEADERS,
                ...(CSRF_METHODS.includes(method) && csrf.tokenHeader),
                ...headers,
                ...jaegerDebugHeader,
                [X_RETPATH_Y]: window.location.href,
                [ETestHeaders.X_YA_PAGE_TOKEN]: pageManager.pageToken ?? '',
            },
            params: {
                srcParams: getStoredSrcParams(),
                ...params,
            },
            ...requestOptions,
        };
    }
}
