import IPrice from 'utilities/currency/PriceInterface';
import {compareWith} from 'projects/avia/lib/comparators/compareWith';
import {ECompareResult} from 'projects/avia/lib/comparators/TComparator';

import {PriceConverter} from './priceConverter';

const cmp = (a: number, b: number): ECompareResult =>
    Math.sign(Math.round(a) - Math.round(b));

export class PriceComparator {
    compare = compareWith<IPrice | undefined>(
        (a?: IPrice, b?: IPrice): ECompareResult => {
            if (!a || !b) {
                return a ? ECompareResult.LT : ECompareResult.GT;
            }

            if (a.currency === b.currency) {
                return cmp(a.value, b.value);
            }

            return cmp(
                this.priceConverter.convertToPreferredCurrency(a).value,
                this.priceConverter.convertToPreferredCurrency(b).value,
            );
        },
    );

    private priceConverter: PriceConverter;

    constructor(priceConverter: PriceConverter) {
        this.priceConverter = priceConverter;

        this.min = this.min.bind(this);
        this.max = this.max.bind(this);
    }

    isLess = (a?: IPrice, b?: IPrice): boolean =>
        this.compare(a, b) === ECompareResult.LT;

    isLessOrEqual = (a?: IPrice, b?: IPrice): boolean => {
        const compareResult = this.compare(a, b);

        return (
            compareResult === ECompareResult.EQ ||
            compareResult === ECompareResult.LT
        );
    };

    isEqualPrice = (a?: IPrice, b?: IPrice) =>
        this.compare(a, b) === ECompareResult.EQ;

    min<T extends IPrice>(a: T, b: T): T;
    min<T extends IPrice>(a: T, b?: T): T;
    min<T extends IPrice>(a: T | undefined, b: T): T;
    min<T extends IPrice>(a?: T, b?: T): T | undefined {
        // eslint-disable-next-line no-nested-ternary
        return this.isLess(a, b) ? (a ? a : b) : b;
    }

    max<T extends IPrice>(a: T, b: T): T;
    max<T extends IPrice>(a: T | undefined, b: T): T;
    max<T extends IPrice>(a: T, b?: T): T;
    max<T extends IPrice>(a?: T, b?: T): T | undefined {
        // eslint-disable-next-line no-nested-ternary
        return this.isLess(a, b) ? (b ? b : a) : a;
    }
}
