import {NOT_ORIGINAL_FINE} from 'constants/currencies';

import {
    TRatesMap,
    ICurrenciesInfo,
} from 'server/services/CurrenciesService/types';

import {CurrencyType} from 'utilities/currency/CurrencyType';
import IPrice from 'utilities/currency/PriceInterface';

interface IConversionParams {
    value: number;
    currency: CurrencyType;
    currenciesInfo: ICurrenciesInfo;
    withFine?: boolean;
    fine?: number;
}

export function getCurrencyRate(
    currency: CurrencyType,
    currencyRates: TRatesMap,
): number | undefined {
    return currencyRates[currency];
}

export function isConversionAvailable(
    currency: CurrencyType,
    currencyRates: TRatesMap,
): boolean {
    return Boolean(getCurrencyRate(currency, currencyRates));
}

export function convertFromCurrency({
    value,
    currency,
    currenciesInfo: {currencyRates},
    withFine,
    fine = NOT_ORIGINAL_FINE,
}: IConversionParams): number {
    const factor = withFine ? fine : 1;
    const rate = getCurrencyRate(currency, currencyRates);

    return rate ? value * (rate / factor) : value;
}

export function convertToCurrency({
    value,
    currency,
    currenciesInfo: {currencyRates},
    withFine,
    fine = NOT_ORIGINAL_FINE,
}: IConversionParams) {
    const factor = withFine ? fine : 1;
    const rate = getCurrencyRate(currency, currencyRates);

    return rate ? value / (rate * factor) : value;
}

export function convertPriceToPreferredCurrency({
    value,
    currency,
    currenciesInfo,
    withFine,
    fine = NOT_ORIGINAL_FINE,
}: IConversionParams): IPrice {
    const {currencyRates, preferredCurrency, nationalCurrency} = currenciesInfo;

    if (
        currency === preferredCurrency ||
        !isConversionAvailable(currency, currencyRates) ||
        !isConversionAvailable(preferredCurrency, currencyRates)
    ) {
        return {
            value,
            currency,
        };
    }

    const valueInNationalCurrency =
        nationalCurrency === currency
            ? value
            : convertFromCurrency({
                  value,
                  currency,
                  currenciesInfo,
              });
    const valueInPreferredCurrency =
        nationalCurrency === preferredCurrency
            ? valueInNationalCurrency
            : convertToCurrency({
                  value: valueInNationalCurrency,
                  currency: preferredCurrency,
                  currenciesInfo,
              });

    return {
        value: withFine
            ? Math.floor(fine * valueInPreferredCurrency)
            : valueInPreferredCurrency,
        currency: preferredCurrency,
    };
}
