import {CurrencyType} from 'utilities/currency/CurrencyType';
import ValueType from 'utilities/currency/ValueType';
import {CHAR_SIX_PER_EM_SPACE} from 'utilities/strings/charCodes';
import formatInteger from 'utilities/numbers/formatInteger';

import * as i18nBlock from 'i18n/common-price';

export const CURRENCY_SYMBOLS = {
    [CurrencyType.RUB]: '₽',
    [CurrencyType.RUR]: '₽',
    [CurrencyType.KZT]: '₸',
    [CurrencyType.USD]: '$',
    [CurrencyType.EUR]: '€',

    [CurrencyType.TRY]: null,
    [CurrencyType.GBP]: null,
    [CurrencyType.BYN]: null,
    [CurrencyType.UAH]: null,
    [CurrencyType.UZS]: null,
};

const FORMATTERS = {
    eastern: (whole: string, cents: string | undefined): string =>
        typeof cents === 'undefined' ? whole : `${whole},${cents}`,
    western: (whole: string, cents: string | undefined): string =>
        typeof cents === 'undefined' ? whole : `${whole}.${cents}`,
};

const FORMATTERS_WITH_SYMBOL = {
    EASTERN: (
        whole: string,
        cents: string | undefined,
        symbol: string,
        isCurrencyShown: boolean,
        postfix: string | undefined,
    ): string =>
        `${FORMATTERS.eastern(whole, cents)}${postfix || ''}${
            isCurrencyShown ? `${CHAR_SIX_PER_EM_SPACE}${symbol}` : ''
        }`,
    WESTERN: (
        whole: string,
        cents: string | undefined,
        symbol: string,
        isCurrencyShown: boolean,
        postfix: string | undefined,
    ): string =>
        `${isCurrencyShown ? symbol : ''}${FORMATTERS.western(whole, cents)}${
            postfix || ''
        }`,
};

const CURRENCY_FORMATTERS_WITH_SYMBOL = {
    [CurrencyType.RUB]: FORMATTERS_WITH_SYMBOL.EASTERN,
    [CurrencyType.RUR]: FORMATTERS_WITH_SYMBOL.EASTERN,
    [CurrencyType.UAH]: FORMATTERS_WITH_SYMBOL.EASTERN,
    [CurrencyType.KZT]: FORMATTERS_WITH_SYMBOL.EASTERN,
    [CurrencyType.BYN]: FORMATTERS_WITH_SYMBOL.EASTERN,
    [CurrencyType.UZS]: FORMATTERS_WITH_SYMBOL.EASTERN,

    [CurrencyType.USD]: FORMATTERS_WITH_SYMBOL.WESTERN,
    [CurrencyType.EUR]: FORMATTERS_WITH_SYMBOL.WESTERN,

    [CurrencyType.GBP]: null,
    [CurrencyType.TRY]: null,
};

/**
 * Форматирует значение цены и подставляет симфол валюты.
 *
 * @param value - Значение.
 * @param currency - Код валюты.
 * @param [isCurrencyShown=true] - Показывать знак валюты.
 * @param [isFrom=false] - Показывать префикс от.
 * @param [isRound=false] - Показывать цифры после запятой.
 * @param [forceCentsVisibility=false] - Нужно ли показывать центы если они равны 00
 * @param postfix - Постфикс для value
 * @param thousandsDelimiter - символ разделителя на тысячи
 */
export function formatPrice({
    value,
    currency,
    isCurrencyShown = true,
    isRound = false,
    isFrom = false,
    forceCentsVisibility = false,
    postfix,
    thousandsDelimiter = CHAR_SIX_PER_EM_SPACE,
}: {
    value: ValueType;
    currency: CurrencyType;
    isCurrencyShown?: boolean;
    isRound?: boolean;
    isFrom?: boolean;
    forceCentsVisibility?: boolean;
    postfix?: string;
    thousandsDelimiter?: string;
}): string {
    const symbol = CURRENCY_SYMBOLS[currency] || currency;
    const formatter =
        CURRENCY_FORMATTERS_WITH_SYMBOL[currency] ||
        FORMATTERS_WITH_SYMBOL.EASTERN;
    const preparedValue = typeof value === 'string' ? parseFloat(value) : value;

    let [whole, cents] = preparedValue.toFixed(isRound ? 0 : 2).split('.') as [
        string,
        string | undefined,
    ];

    whole = formatInteger(whole, thousandsDelimiter);

    if (cents === '00' && !forceCentsVisibility) {
        cents = undefined;
    }

    const price = formatter(whole, cents, symbol, isCurrencyShown, postfix);

    if (!isFrom) {
        return price;
    }

    return i18nBlock._from({price});
}
