import {NOT_ORIGINAL_FINE} from 'constants/currencies';

import {ICurrenciesInfo} from 'server/services/CurrenciesService/types';

import {DEFAULT_CURRENCIES} from 'reducers/common/currencies/reducer';

import IPrice from 'utilities/currency/PriceInterface';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import {convertPriceToPreferredCurrency} from 'utilities/currency/convertPrice';

interface IPriceConverterOptions {
    currenciesInfo: ICurrenciesInfo;
    withFine?: boolean;
    fine?: number;
}

const RUSSIAN_CURRENCIES = new Set([CurrencyType.RUR, CurrencyType.RUB]);

export class PriceConverter {
    private currenciesInfo: ICurrenciesInfo = DEFAULT_CURRENCIES;
    private withFine: boolean = true;
    private fine: number = NOT_ORIGINAL_FINE;

    constructor(options?: IPriceConverterOptions) {
        if (options) {
            this.setOptions(options);
        }
    }

    setOptions(options: IPriceConverterOptions): void {
        this.currenciesInfo = options.currenciesInfo;

        if (options.withFine !== undefined) {
            this.withFine = options.withFine;
        }

        if (options.fine !== undefined) {
            this.fine = options.fine;
        }
    }

    convertToPreferredCurrency({value, currency}: IPrice): IPrice {
        return convertPriceToPreferredCurrency({
            value,
            currency,
            currenciesInfo: this.currenciesInfo,
            withFine: this.withFine,
            fine: this.fine,
        });
    }

    isRussianCurrency(currency: CurrencyType): boolean {
        return RUSSIAN_CURRENCIES.has(currency);
    }

    // Бекенд авиа возвращает цены в RUR, другие бекенды в RUB
    // но мы должны считать их одной и той же валютой
    areSameCurrencies(first: CurrencyType, second: CurrencyType): boolean {
        return (
            first === second ||
            (this.isRussianCurrency(first) && this.isRussianCurrency(second))
        );
    }

    isPreferredCurrency(currency: CurrencyType): boolean {
        return this.areSameCurrencies(
            currency,
            this.currenciesInfo.preferredCurrency,
        );
    }

    getPreferredCurrency(): CurrencyType {
        return this.currenciesInfo.preferredCurrency;
    }
}
