import {useCallback, useReducer} from 'react';

enum EActionType {
    RESET = 'reset',
    LOADING = 'loading',
    SUCCESS = 'success',
    ERROR = 'error',
}

interface IAsyncState {
    isLoading: boolean;
    isSuccess: boolean;
    isError: boolean;
}

export const ASYNC_INITIAL_STATE: IAsyncState = {
    isLoading: false,
    isSuccess: false,
    isError: false,
};

export const ASYNC_LOADING_STATE: IAsyncState = {
    isLoading: true,
    isSuccess: false,
    isError: false,
};

export const ASYNC_SUCCESS_STATE: IAsyncState = {
    isLoading: false,
    isSuccess: true,
    isError: false,
};

export const ASYNC_ERROR_STATE: IAsyncState = {
    isLoading: false,
    isSuccess: false,
    isError: true,
};

function reducer(_: IAsyncState, action: EActionType): IAsyncState {
    const mapping = {
        [EActionType.RESET]: ASYNC_INITIAL_STATE,
        [EActionType.LOADING]: ASYNC_LOADING_STATE,
        [EActionType.SUCCESS]: ASYNC_SUCCESS_STATE,
        [EActionType.ERROR]: ASYNC_ERROR_STATE,
    };

    return mapping[action];
}

export function useAsyncState(defaultState: IAsyncState = ASYNC_INITIAL_STATE) {
    const [state, dispatch] = useReducer(reducer, defaultState);

    const loading = useCallback(() => dispatch(EActionType.LOADING), []);
    const success = useCallback(() => dispatch(EActionType.SUCCESS), []);
    const error = useCallback(() => dispatch(EActionType.ERROR), []);
    const reset = useCallback(() => dispatch(EActionType.RESET), []);

    return {
        ...state,
        loading,
        success,
        error,
        reset,
    };
}
