import {RefObject, useEffect, useRef} from 'react';

export function useEventListener(
    eventName: keyof WindowEventMap,
    handler: (ev: Event) => void,
    ref?: RefObject<HTMLElement>,
    options?: AddEventListenerOptions,
): void {
    const savedHandler = useRef<(ev: Event) => void>();

    // Update ref.current value if handler changes.
    // This allows our effect below to always get latest handler ...
    // ... without us needing to pass it in effect deps array ...
    // ... and potentially cause effect to re-run every render.
    useEffect(() => {
        savedHandler.current = handler;
    }, [handler]);

    useEffect(() => {
        const target = ref?.current || window;
        const eventListener = (event: Event): void =>
            savedHandler.current?.(event);

        target.addEventListener(eventName, eventListener, options);

        return (): void => {
            target.removeEventListener(eventName, eventListener, options);
        };
    }, [eventName, ref, options]);
}
