import {useRef} from 'react';

type TValueOfParams<Params extends any[]> = Params extends (infer T)[]
    ? T
    : never;

type TParametersMap<Callback extends (...args: any[]) => any> = Map<
    TValueOfParams<Parameters<Callback>> | Callback,
    TParametersMap<Callback>
>;

type TCacheMap<Callback extends (...args: any[]) => any> = WeakMap<
    Callback,
    TParametersMap<Callback>
>;

/**
 * Возвращает кешированную обертку над функцией callback, которая будет выполнена с определенными параметрами.
 *
 * Позволяет не терять типизацию параметров при использовании в компонентах, возвращая одну и ту
 * же функцию при многократных вызовах.
 *
 * Пример использования:
 *
 * const getCallback = useGetCachedCallback((numberOfElement: number) => {
 *     clickOnElement(numberOfElement)
 * });
 *
 * const components = elements.map((elementText, index) => (
 *  <span onClick={getCallback(index)}>{elementText}</span>
 * ));
 */
export default function useGetCachedCallback<
    Callback extends (...args: any[]) => any,
>(
    callback: Callback,
): (...args: Parameters<Callback>) => () => ReturnType<Callback> {
    const cacheObject = useRef<TCacheMap<Callback>>(new WeakMap());

    const mapForCallback =
        cacheObject.current.get(callback) ||
        cacheObject.current.set(callback, new Map()).get(callback)!;

    return (...params: Parameters<Callback>): (() => ReturnType<Callback>) =>
        params.reduce<any>((acc, param, index) => {
            return (
                acc.get(param) ||
                acc
                    .set(
                        param,
                        index === params.length - 1
                            ? () =>
                                  // eslint-disable-next-line no-useless-call,prefer-spread
                                  callback.apply(null, params)
                            : new Map(),
                    )
                    .get(param)
            );
        }, mapForCallback);
}
