import {useEffect, useState} from 'react';
import axios from 'axios';

import {delay} from 'utilities/async/delay';
import {useAsyncState} from 'utilities/hooks/useAsyncState';

import {browserHotelBookProvider} from 'serviceProvider/hotels/book/browserHotelBookProvider';

const POLLING_DELAY_TIME = 3000;

interface IVoucherStatus {
    isReady: boolean;
    isLoading: boolean;
    isError: boolean;
}

export function useHotelVoucherStatus(orderId: string): IVoucherStatus {
    const {isLoading, isError, loading, success, error} = useAsyncState();
    const [isReady, setIsReady] = useState(false);

    useEffect(() => {
        const abortController = new AbortController();

        (async (): Promise<void> => {
            loading();

            try {
                // eslint-disable-next-line no-constant-condition
                while (true) {
                    const {voucherIsReady} =
                        await browserHotelBookProvider.checkVoucher(
                            {orderId},
                            {signal: abortController.signal},
                        );

                    if (voucherIsReady) {
                        setIsReady(true);

                        break;
                    }

                    await delay(POLLING_DELAY_TIME);
                }

                success();
            } catch (err) {
                if (!axios.isCancel(err)) {
                    console.log(err);

                    error();
                }
            }
        })();

        return (): void => {
            abortController.abort();
        };
    }, [error, loading, orderId, success]);

    return {isReady, isLoading, isError};
}
