import {useCallback} from 'react';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

export enum EShareStatus {
    // Браузер не поддерживает вообще
    UNSUPPORTED = 'unsupported',
    // Web share API есть, но параметры некорректные
    INVALID_DATA = 'invalid_data',

    // Ошибка, но стоит учесть, что отмена "поделиться" выкидывает AbortError
    FAILED = 'failed',

    OK = 'ok',
}

export interface IShareViaNativeAPIResponse {
    status: EShareStatus;
    error?: Error;
}

type TUseNativeShareAPICbType = (
    shareData?: ShareData,
) => Promise<IShareViaNativeAPIResponse>;

export const useNativeShareAPI = (): TUseNativeShareAPICbType => {
    const deviceType = useDeviceType();

    return useCallback(
        async (shareData?: ShareData): Promise<IShareViaNativeAPIResponse> => {
            const hasSupport =
                typeof navigator !== 'undefined' &&
                typeof navigator.share === 'function';

            if (!hasSupport || deviceType.isDesktop) {
                return {status: EShareStatus.UNSUPPORTED};
            }

            const canShare = navigator.canShare(shareData);

            if (!canShare) {
                return {status: EShareStatus.INVALID_DATA};
            }

            try {
                await navigator.share(shareData);

                return {status: EShareStatus.OK};
            } catch (e) {
                return {status: EShareStatus.FAILED, error: e};
            }
        },
        [deviceType],
    );
};
