import {useDispatch, useSelector} from 'react-redux';
import pathToRegexp from 'path-to-regexp';
import {useHistory} from 'react-router-dom';

import {EProjectName} from 'constants/common';
import {URLs} from 'constants/urls';

import {ITransformSearchContextRequestParams} from 'server/api/TransformSearchApi/types/ITransformSearchContextRequestParams';
import {ITransformSearchContextResponse} from 'server/api/TransformSearchApi/types/ITransformSearchContextResponse';
import {EPubSubEvent} from 'types/EPubSubEvent';
import {ESearchFormFieldName} from 'components/SearchForm/types';

import {transformSearchContext as aviaTransformSearchContext} from 'reducers/avia/searchForm/thunk/transformSearchContext';
import {transformSearchContext as trainsTransformSearchContext} from 'reducers/trains/searchForm/thunk/transformSearchContext';
import {transformSearchContext as hotelsTransformSearchContext} from 'reducers/hotels/searchForm/thunk/transformSearchContext';
import {transformSearchContext as busesTransformSearchContext} from 'reducers/buses/searchForm/thunk/transformSearchContext';
import {CustomDispatch} from 'reducers/trains/customDispatch';

import crossSearchInfoSelector from './selectors/crossSearchInfoSelector';

import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {publishSync, subscribeOnce} from 'utilities/pubSub';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getWhenMoment from 'utilities/dateUtils/when/getWhenMoment';
import {ROBOT} from 'utilities/dateUtils/formats';

type TTransformSearchContextParams = Omit<
    ITransformSearchContextRequestParams,
    'vertical'
>;

interface IVerticalCrossSearchInfo {
    getCrossSearchParams(): TTransformSearchContextParams | undefined;
    onLinkClick(
        params: TTransformSearchContextParams,
    ): Promise<ITransformSearchContextResponse | null>;
}

const aviaIndexRegExp = pathToRegexp(URLs.aviaRoot);
const aviaSearchRootRegExp = pathToRegexp(URLs.aviaSearchResults);
const trainsIndexRegExp = pathToRegexp(URLs.trainsRoot);
const trainsSearchRootRegExp = pathToRegexp(URLs.trainsSearch);
const hotelsIndexRegExp = pathToRegexp(URLs.hotelsRoot);
const hotelsSearchRootRegExp = pathToRegexp(URLs.hotelsSearch);
const busesIndexRegExp = pathToRegexp(URLs.busesRoot);
const busesSearchRootRegExp = pathToRegexp(URLs.busesSearch);

// Пока интересует только переход со страниц поиска и морды
const getProjectByPathname = (pathname: string): EProjectName | undefined => {
    if (pathname === URLs[EProjectName.INDEX]) {
        return EProjectName.HOTELS;
    }

    if (aviaIndexRegExp.test(pathname) || aviaSearchRootRegExp.test(pathname)) {
        return EProjectName.AVIA;
    }

    if (
        trainsIndexRegExp.test(pathname) ||
        trainsSearchRootRegExp.test(pathname)
    ) {
        return EProjectName.TRAINS;
    }

    if (
        hotelsIndexRegExp.test(pathname) ||
        hotelsSearchRootRegExp.test(pathname)
    ) {
        return EProjectName.HOTELS;
    }

    if (
        busesIndexRegExp.test(pathname) ||
        busesSearchRootRegExp.test(pathname)
    ) {
        return EProjectName.BUSES;
    }
};

export const useOnNavigateToProject = (): ((
    targetProject: EProjectName,
) => void) => {
    const dispatch: CustomDispatch = useDispatch();
    const history = useHistory();
    const {isDesktop} = useDeviceType();

    const {
        aviaFromField,
        aviaToField,
        aviaStartDate,
        aviaEndDate,
        trainsFromField,
        trainsToField,
        trainsStartDate,
        trainsEndDate,
        hotelsToField,
        busesFromField,
        busesToField,
        busesStartDate,
    } = useSelector(crossSearchInfoSelector);

    const crossSearchInfo: PartialRecord<
        EProjectName,
        IVerticalCrossSearchInfo
    > = {
        [EProjectName.AVIA]: {
            getCrossSearchParams(): TTransformSearchContextParams | undefined {
                if (
                    !aviaFromField.selectedValue ||
                    !aviaToField.selectedValue
                ) {
                    return;
                }

                return {
                    fromPointKey: aviaFromField.selectedValue.pointKey,
                    toPointKey: aviaToField.selectedValue.pointKey,
                    dateForward: aviaStartDate ?? undefined,
                    dateBackward: aviaEndDate ?? undefined,
                };
            },
            async onLinkClick(
                params,
            ): Promise<ITransformSearchContextResponse | null> {
                // Заполняем только если какое-то поле пустое
                if (aviaFromField.selectedValue && aviaToField.selectedValue) {
                    return null;
                }

                return dispatch(aviaTransformSearchContext(params));
            },
        },
        [EProjectName.TRAINS]: {
            getCrossSearchParams(): TTransformSearchContextParams | undefined {
                if (
                    !trainsFromField.selectedValue ||
                    !trainsToField.selectedValue
                ) {
                    return;
                }

                return {
                    fromPointKey: trainsFromField.selectedValue.pointKey,
                    toPointKey: trainsToField.selectedValue.pointKey,
                    dateForward: trainsStartDate
                        ? getWhenMoment(trainsStartDate).format(ROBOT)
                        : undefined,
                    dateBackward: trainsEndDate
                        ? getWhenMoment(trainsEndDate).format(ROBOT)
                        : undefined,
                };
            },
            async onLinkClick(
                params,
            ): Promise<ITransformSearchContextResponse | null> {
                // Заполняем только если какое-то поле пустое
                if (
                    trainsFromField.selectedValue &&
                    trainsToField.selectedValue
                ) {
                    return null;
                }

                return dispatch(trainsTransformSearchContext(params));
            },
        },
        [EProjectName.HOTELS]: {
            getCrossSearchParams(): TTransformSearchContextParams | undefined {
                if (!hotelsToField.selectedValue) {
                    return;
                }

                const {geoId} = hotelsToField.selectedValue.redirectParams;

                if (!geoId) {
                    return;
                }

                return {
                    toGeoId: geoId,
                };
            },
            async onLinkClick(
                params,
            ): Promise<ITransformSearchContextResponse | null> {
                // Заполняем только если поле "Куда" пустое
                if (hotelsToField.selectedValue) {
                    return null;
                }

                return dispatch(hotelsTransformSearchContext(params));
            },
        },
        [EProjectName.BUSES]: {
            getCrossSearchParams(): TTransformSearchContextParams | undefined {
                if (
                    !busesFromField.selectedValue ||
                    !busesToField.selectedValue
                ) {
                    return;
                }

                return {
                    fromPointKey: busesFromField.selectedValue.pointKey,
                    toPointKey: busesToField.selectedValue.pointKey,
                    dateForward: busesStartDate
                        ? getWhenMoment(busesStartDate).format(ROBOT)
                        : undefined,
                };
            },
            async onLinkClick(
                params,
            ): Promise<ITransformSearchContextResponse | null> {
                // Заполняем только если какое-то поле пустое
                if (
                    busesFromField.selectedValue &&
                    busesToField.selectedValue
                ) {
                    return null;
                }

                return dispatch(busesTransformSearchContext(params));
            },
        },
    };

    return useImmutableCallback(async targetProject => {
        const currentProject = getProjectByPathname(history.location.pathname);

        if (!currentProject) {
            return;
        }

        const currentProjectCrossSearchInfo = crossSearchInfo[currentProject];
        const targetProjectCrossSearchInfo = crossSearchInfo[targetProject];

        if (!currentProjectCrossSearchInfo || !targetProjectCrossSearchInfo) {
            return;
        }

        const params = currentProjectCrossSearchInfo.getCrossSearchParams();

        if (!params) {
            return;
        }

        const [crossSearchResponse] = await Promise.all([
            targetProjectCrossSearchInfo.onLinkClick(params),
            new Promise<void>(resolve => {
                subscribeOnce(EPubSubEvent.SEARCH_FORM_MOUNT, resolve);
            }),
        ]);

        // Фокусируем поле с датой на десктопе, если пункты заполнились, а дата нет
        // Для проброса из/в отели игнорируем заполненность обоих полей from/to
        // Для проброса в отели также игнорируем наличие/отсутствие даты в респонсе
        if (
            isDesktop &&
            crossSearchResponse &&
            (currentProject === EProjectName.HOTELS ||
                crossSearchResponse.from) &&
            (targetProject === EProjectName.HOTELS ||
                (crossSearchResponse.to && !crossSearchResponse.dateForward))
        ) {
            publishSync(
                EPubSubEvent.FOCUS_SEARCH_FORM_FIELD,
                ESearchFormFieldName.START_DATE,
            );
        }
    });
};
