import {useEffect, useMemo, useRef, useState} from 'react';
import {useSelector} from 'react-redux';
import pick from 'lodash/pick';
import isEqual from 'lodash/isEqual';

import {TQueryType, TQueryTypeWithArray} from 'types/common/IQueryParams';

import entryPageSelector from 'selectors/common/entryPageSelector';

import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import browserHistory from 'utilities/browserHistory/browserHistory';

export default function useQuery<Key extends string>(
    queries: Key[] | undefined,
    allowArrays: true,
): PartialRecord<Key, TQueryTypeWithArray>;
export default function useQuery<Key extends string>(
    queries?: Key[],
    allowArrays?: false,
): PartialRecord<Key, TQueryType>;

export default function useQuery<Key extends string>(
    queries?: Key[],
    allowArrays?: boolean,
): PartialRecord<Key, TQueryType | TQueryTypeWithArray> {
    const entryPageUrl = useSelector(entryPageSelector);

    const getPickedQuery = useImmutableCallback(
        (location: {search: string}) => {
            const allQuery = getQueryByLocation(location, allowArrays ?? false);

            return (
                queries ? pick(allQuery, queries) : allQuery
            ) as PartialRecord<Key, TQueryTypeWithArray>;
        },
    );

    const pickedQueryRef = useRef(
        getPickedQuery(
            browserHistory
                ? browserHistory.location
                : {search: entryPageUrl?.split('?')[1] || ''},
        ),
    );
    const [pickedQuery, setPickedQuery] = useState(pickedQueryRef.current);

    const prevQueriesRef = useRef(queries);

    const memoQueries = useMemo(() => {
        if (isEqual(prevQueriesRef.current, queries)) {
            return prevQueriesRef.current;
        }

        prevQueriesRef.current = queries;

        return prevQueriesRef.current;
    }, [queries]);

    useEffect(() => {
        if (!browserHistory) {
            return;
        }

        const unsubscribe = browserHistory.listen(location => {
            const newPickedQuery = getPickedQuery(location);

            if (isEqual(newPickedQuery, pickedQueryRef.current)) {
                return;
            }

            setPickedQuery(newPickedQuery);
            pickedQueryRef.current = newPickedQuery;
        });

        return (): void => {
            unsubscribe();
        };
    }, [allowArrays, getPickedQuery, memoQueries]);

    return pickedQuery;
}
