import {useEffect, useState, useCallback} from 'react';

const USER_INTERACTION_EVENTS = ['mousemove', 'mousedown', 'keydown'];

/**
 * Хук который, который вызовет колбек по таймауту при неактивности пользователя.
 *
 * NOTE!: при изменении параметров передаваемых в аргументы, таймаут будет
 * перезапущен
 *
 * @param cb - колбек, который необхедимо вызвать при неактивности пользователя
 * @param timeOut - время неактивности в мс
 *
 * @example
 * ```tsx
 * // setIsInactive(true) сработает если пользователь будет неактивен 60 секунд
 * useUserInactivityCallback(useCallback(() => setIsInactive(true)), 60000);
 * ```
 */
export function useUserInactivityCallback(
    cb: (...arg: any[]) => void,
    timeOut: number,
): void {
    const [interactionId, setInteractionId] = useState(0);

    useEffect(() => {
        const handleId = setTimeout(cb, timeOut);

        return () => {
            clearTimeout(handleId);
        };
    }, [cb, timeOut, interactionId]);

    const handleInteraction = useCallback(
        () => setInteractionId(x => x + 1),
        [],
    );

    useEffect(() => {
        USER_INTERACTION_EVENTS.forEach(eventName => {
            document.body.addEventListener(eventName, handleInteraction);
        });

        return () => {
            USER_INTERACTION_EVENTS.forEach(eventName => {
                document.body.removeEventListener(eventName, handleInteraction);
            });
        };
    }, [handleInteraction]);
}
