import {IImage} from 'types/common/IImage';

/**
 * Возвращает оптимальное изображение для указанной высоты и ширины в пикселях.
 *
 * Алгоритм попытается подобрать наиболее близкое, к указанным размерам изображение,
 * стараясь выбрать в первую очередь такое, которое не меньше указанных параметров.
 */
export function getOptimalImage(
    images: IImage[],
    width: number,
    height: number,
): IImage | undefined {
    if (images.length === 0) {
        return undefined;
    }

    return images.reduce((a, b) => {
        const aSuitable = a.height >= height && a.width >= width;
        const bSuitable = b.height >= height && b.width >= width;

        // Если только одно изображение подходит под параметры, то возвращаем его
        if (aSuitable && !bSuitable) {
            return a;
        }

        if (!aSuitable && bSuitable) {
            return b;
        }

        const aSize = a.width * a.height;
        const bSize = b.width * b.height;

        // Если оба изображения подходят, то выбираем самое маленькое
        if (aSuitable && bSuitable) {
            return aSize < bSize ? a : b;
        }

        const aHeightRatio = a.height / height;
        const aWidthResized = a.width / aHeightRatio;
        const aResizedSize =
            (a.height < height ? a.height : height) *
            (aWidthResized < width ? aWidthResized : width);

        const bHeightRatio = b.height / height;
        const bWidthResized = b.width / bHeightRatio;
        const bResizedSize =
            (b.height < height ? b.height : height) *
            (bWidthResized < width ? bWidthResized : width);

        /**
         * Если изображения не подходят под параметры, то выбираем то,
         * что заполнит больше всего пикселей при вписывании в квадрат из параметров,
         * с учетом масштабирования по высоте
         */
        if (!aSuitable && !bSuitable) {
            return aResizedSize > bResizedSize ? a : b;
        }

        return a;
    });
}
