import Cookie from 'js-cookie';

import {PAGE_API_REQUESTS_ENABLED} from 'projects/testControlPanel/pages/TestRequestManagerPage/constants';

import {ISocketRequestParams} from 'server/utilities/TestRequestManager/types/connectParams';

import browserHistory from 'utilities/browserHistory/browserHistory';
import {formatQuery} from 'utilities/url';

export interface IPageNotLoadedInfo {
    loaded: false;
    loadDuration: null;
    loadedAt: null;
}

export interface IPageLoadedInfo {
    loaded: true;
    loadDuration: number;
    loadedAt: number;
}

export type TPageLoadInfo = IPageNotLoadedInfo | IPageLoadedInfo;

export type TOnLoadCallback = (pageLoadInfo: IPageLoadedInfo) => unknown;
export type TOnFirstNavigationCallback = () => unknown;
export type TUnregisterCallback = () => void;

export interface IInitOptions {
    isProductionEnv: boolean;
    pageToken: string | null;
}

class PageManager {
    pageToken: string | null = null;
    pageLoadInfo: TPageLoadInfo = {
        loaded: false,
        loadDuration: null,
        loadedAt: null,
    };
    wasNavigation: boolean = false;
    apiRequestsSocket: WebSocket | null = null;

    private initialized: boolean = false;
    private initialPagePathname: string | undefined;
    private isProductionEnv: boolean = true;
    private loadCallbacks: Set<TOnLoadCallback> = new Set();
    private firstNavigationCallbacks: Set<TOnFirstNavigationCallback> =
        new Set();

    addOnLoadCallback(callback: TOnLoadCallback): TUnregisterCallback {
        this.loadCallbacks.add(callback);

        return (): void => {
            this.loadCallbacks.delete(callback);
        };
    }

    addOnFirstNavigationCallback(
        callback: TOnFirstNavigationCallback,
    ): TUnregisterCallback {
        this.firstNavigationCallbacks.add(callback);

        return (): void => {
            this.firstNavigationCallbacks.delete(callback);
        };
    }

    async init(options: IInitOptions): Promise<void> {
        if (this.initialized) {
            return;
        }

        this.initialized = true;
        this.initialPagePathname = browserHistory?.location.pathname;
        this.isProductionEnv = options.isProductionEnv;
        this.pageToken = options.pageToken;

        /**
         * Ждем открытия сокета для того, чтобы успели зарегистрироваться все события о запросах
         */
        if (
            !this.isProductionEnv &&
            Cookie.get(PAGE_API_REQUESTS_ENABLED) === 'true'
        ) {
            const {socket, opened} = this.initApiRequestsSocket(true);

            await opened;

            this.apiRequestsSocket = socket;
        }

        const removeListener = browserHistory?.listen(location => {
            if (location.pathname === this.initialPagePathname) {
                return;
            }

            removeListener?.();

            this.wasNavigation = true;

            this.firstNavigationCallbacks.forEach(callback => {
                callback();
            });
        });

        if (document.readyState === 'complete') {
            this.onLoad();

            return;
        }

        document.addEventListener('readystatechange', () => {
            if (document.readyState === 'complete') {
                this.onLoad();
            }
        });
    }

    initApiRequestsSocket(filterByPage: boolean): {
        socket: WebSocket;
        opened: Promise<void>;
    } {
        const query: ISocketRequestParams = {
            pageToken: filterByPage ? this.pageToken ?? undefined : undefined,
        };
        const queryString = formatQuery(query, {
            filterNull: true,
        });
        const socket = new WebSocket(
            `wss://${location.host}/api/testControlPanel/apiRequestsChannel?${queryString}`,
        );

        return {
            socket,
            opened: new Promise<void>(resolve => {
                socket.addEventListener('open', () => resolve());
            }),
        };
    }

    private onLoad(): void {
        this.pageLoadInfo = {
            loaded: true,
            loadDuration: performance.now(),
            loadedAt: Date.now(),
        };

        for (const loadCallback of this.loadCallbacks) {
            loadCallback(this.pageLoadInfo);
        }
    }
}

export default new PageManager();
