import axios, {AxiosInstance, AxiosResponse} from 'axios';
import toSnakeCase from 'lodash/snakeCase';
import moment from 'moment';

import {BASE_AXIOS_HEADERS} from 'constants/axios';

import {IDocumentTypesDTO} from 'server/api/TravelersApi/types/IDocumentTypesDTO';
import {IPassengerWithDocumentsAndBonusCardsDTO} from 'server/api/TravelersApi/types/IPassengerDTO';
import {IPassengerDocumentData} from 'types/common/document/IPassengerDocumentData';
import {IDocumentDTO} from 'server/api/TravelersApi/types/IDocumentDTO';

import {IPreparedPassenger} from 'reducers/account/passengers/api-types';

import {ROBOT, HUMAN_DATE_RU} from 'utilities/dateUtils/formats';
import {csrf} from 'utilities/csrfToken/createGlobalTokenGetter';
import TDataAddPassenger from 'utilities/passengerApiMethods/TDataAddPassenger';

function createAxios(): AxiosInstance {
    return axios.create({
        baseURL: '/api/account',
        headers: {
            ...BASE_AXIOS_HEADERS,
            ...csrf.tokenHeader,
        },
    });
}

interface IPrepareDataInput {
    expirationDate?: moment.MomentInput;
    birthDate?: moment.MomentInput;

    itn?: string;
    email?: string;
    phone?: string;
    phone_additional?: string;

    // Любой другой тип тоже возможен
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    [key: string]: any;
}

const prepareData = (data: IPrepareDataInput): Record<string, unknown> => {
    const result: Record<string, unknown> = {};

    Object.keys(data).forEach(key => {
        if (key === 'expirationDate' || key === 'birthDate') {
            if (data[key] === '') {
                result[toSnakeCase(key)] = null;
            } else {
                result[toSnakeCase(key)] = moment(
                    data[key],
                    HUMAN_DATE_RU,
                ).format(ROBOT);
            }
        } else if (
            ['itn', 'email', 'phone', 'phone_additional'].includes(key)
        ) {
            result[toSnakeCase(key)] = data[key] || null;
        } else if (key === 'train_notifications_enabled') {
            result[toSnakeCase(key)] = Boolean(data[key]);
        } else {
            result[toSnakeCase(key)] = data[key];
        }
    });

    return result;
};

export const addDocument = (
    data: IPassengerDocumentData,
): Promise<AxiosResponse<IDocumentDTO>> =>
    createAxios().post('/documentAdd', {...prepareData(data)});

export const editDocument = (
    data: IPassengerDocumentData,
): Promise<AxiosResponse<IDocumentDTO>> =>
    createAxios().put('/documentEdit', {...prepareData(data)});

export const removeDocument = (data: {
    id: string;
    passenger_id: string;
}): Promise<AxiosResponse<void>> =>
    createAxios().delete('/documentRemove', {data: prepareData(data)});

// ADDITION OF PASSENGER
interface IAddPassengerReturnTuple {
    passenger: IPassengerWithDocumentsAndBonusCardsDTO;
    document: IDocumentDTO;
}

export const addPassenger = async (
    data: TDataAddPassenger,
): Promise<IAddPassengerReturnTuple> => {
    const result: AxiosResponse<IPassengerWithDocumentsAndBonusCardsDTO> =
        await createAxios().post('/passengerAdd', {
            ...prepareData(data.passenger),
        });

    const passengerData = result.data;

    data.document.passengerId = result.data.id;

    const docResult = await addDocument(data.document);
    const documentData = docResult.data;

    return {passenger: passengerData, document: documentData};
};

export const editPassenger = (
    data: IPreparedPassenger,
): Promise<AxiosResponse<IPassengerWithDocumentsAndBonusCardsDTO>> =>
    createAxios().put('/passengerEdit', {...prepareData(data)});

export const removePassenger = (id: string): Promise<AxiosResponse<void>> =>
    createAxios().delete('/passengerRemove', {data: {id}});

export const fetchDocumentTypes = (): Promise<
    AxiosResponse<IDocumentTypesDTO>
> => createAxios().get('/docTypesGet');
