import {Required} from 'utility-types';

import appData from 'utilities/appData/appData';

/* Constants */

const QA_ATTRIBUTE_NAME = 'data-qa';
const EMPTY_QA_VALUE = undefined;
const EMPTY_QA_OBJECT = {};
const JOIN_QA_SYMBOL = '-';

/* Types */

export interface IWithDataQaAttribute {
    [QA_ATTRIBUTE_NAME]?: string;
}

export type TWithCustomDataQaAttribute = Record<TQaAttributeName, TQaValue>;

export interface IWithQaAttributes extends IWithDataQaAttribute {}

interface IPartialQAObject {
    /**
     * Ключ элемента в списке
     */
    key?: string | number;
    /**
     * qa родителя
     */
    parent?: TQaProps;
    /**
     * qa текущего элемента
     */
    current?: TQaProps;
}

type TObjectQAValue =
    | Required<IPartialQAObject, 'parent'>
    | Required<IPartialQAObject, 'current'>;

export type TQaValue = string | undefined;
export type TQaAttributeName = string;
export type TQaProps = string | undefined | TObjectQAValue | IWithQaAttributes;

/* Utilities */

export function prepareQaAttributes(qaProps: TQaProps): IWithQaAttributes {
    const qaValue = getQa(qaProps);

    return buildQaObject(qaValue, QA_ATTRIBUTE_NAME);
}

function joinQaValues(qaValues: TQaValue[]): TQaValue {
    return qaValues.filter(Boolean).join(JOIN_QA_SYMBOL);
}

export function getQa(qaProps: TQaProps): TQaValue {
    if (appData.isProductionEnv) {
        return EMPTY_QA_VALUE;
    }

    if (instanceOfStringQaValue(qaProps)) {
        return qaProps;
    }

    if (instanceOfDataQaAttribute(qaProps)) {
        return qaProps[QA_ATTRIBUTE_NAME];
    }

    if (instanceOfObjectQaValue(qaProps)) {
        const {key, parent, current} = qaProps;
        const qaKey = typeof key === 'undefined' ? undefined : String(key);
        const qaParent = getQa(parent);
        const qaCurrent = getQa(current);

        return joinQaValues([qaKey, qaParent, qaCurrent]);
    }

    return EMPTY_QA_VALUE;
}

function instanceOfObjectQaValue(qaProps: TQaProps): qaProps is TObjectQAValue {
    return (
        typeof qaProps === 'object' &&
        ('parent' in qaProps || 'current' in qaProps)
    );
}

function instanceOfDataQaAttribute(
    qaProps: TQaProps,
): qaProps is IWithDataQaAttribute {
    return typeof qaProps === 'object' && QA_ATTRIBUTE_NAME in qaProps;
}

function instanceOfStringQaValue(qaProps: TQaProps): qaProps is string {
    return typeof qaProps === 'string';
}

function buildQaObject(
    qaValue: TQaValue,
    qaAttributeName: TQaAttributeName = QA_ATTRIBUTE_NAME,
): IWithQaAttributes | TWithCustomDataQaAttribute {
    if (appData.isProductionEnv || !qaValue) {
        return EMPTY_QA_OBJECT;
    }

    return {
        [qaAttributeName]: qaValue,
    };
}

export function prepareCustomQaAttributes(
    qaAttributeName: TQaAttributeName,
    qaValue: TQaValue,
): IWithQaAttributes | TWithCustomDataQaAttribute {
    return buildQaObject(qaValue, qaAttributeName);
}
