import _get from 'lodash/get';

import {
    TDependentValidations,
    IDependencyCondition,
    TValidations,
    IValidationOptions,
    IFormValidationData,
    TDependentValue,
    EDependentConditionType,
} from 'types/common/validation/validation';
import {IFormValues} from 'types/common/validation/form';

import checkDependentValue from 'utilities/validation/form/checkDependentValue';

const UNIQUE_VALUE = Symbol('filterDependsValue');

function checkFormGroupValue(
    formGroup: any,
    fieldName: string,
    validationRules: TDependentValue,
    formData: IFormValidationData,
): boolean {
    const fieldValue = _get(formGroup, fieldName, UNIQUE_VALUE);

    return fieldValue === UNIQUE_VALUE
        ? false
        : checkDependentValue(fieldValue, validationRules, formData);
}

/** Возвращет true если зависимая валидация должна применяться */
function checkValuesCondition(
    groupValues: IFormValues,
    conditions: IDependencyCondition[],
    formData: IFormValidationData,
): boolean {
    const isMatchConditions = conditions.every(condition => {
        const {path, value: validationRules} = condition;
        const {fieldGroupId, fieldName, type} = path;

        const formGroup = fieldGroupId
            ? _get(formData.formValues, fieldGroupId)
            : groupValues;

        if (Array.isArray(formGroup)) {
            if (type === EDependentConditionType.EVERY) {
                return formGroup.every(group => {
                    return checkFormGroupValue(
                        group,
                        fieldName,
                        validationRules,
                        formData,
                    );
                });
            }

            return formGroup.some(group => {
                return checkFormGroupValue(
                    group,
                    fieldName,
                    validationRules,
                    formData,
                );
            });
        }

        return checkFormGroupValue(
            formGroup,
            fieldName,
            validationRules,
            formData,
        );
    });

    return isMatchConditions;
}

function isValidation(
    validation: TValidations | undefined,
): validation is TValidations {
    return Boolean(validation);
}

/**
 * Возвращает список зависимых валидаций, на основе текущих значений формы
 */
function filterDependentValidations(
    groupValues: IFormValues,
    validations: TDependentValidations,
    formData: IFormValidationData,
    options: IValidationOptions,
): TValidations[] {
    if (!validations) {
        return [];
    }

    const filteredValidations = validations
        .filter(validation =>
            checkValuesCondition(groupValues, validation.conditions, formData),
        )
        .map(validation =>
            options.isSubmit
                ? validation.validation.submit
                : validation.validation.blur,
        )
        .filter(isValidation);

    return filteredValidations;
}

export default filterDependentValidations;
