import moment from 'moment';
import _get from 'lodash/get';

import {
    IDateOffsetFromField,
    IFormValidationData,
} from 'types/common/validation/validation';

import {DateLikeType} from 'utilities/dateUtils';
import {HUMAN_DATE_RU} from 'utilities/dateUtils/formats';
import getDateWithOffset from 'utilities/dateUtils/getDateWithOffset';

/**
 * Проверяет на максимальную дату относительно значения поля
 *
 * @param dateOffset - сдвиг даты относительно поля
 * @param value - значение поля, в формате dd.mm.yyyy
 * @param formData - данные формы
 *
 * @returns true - если значение валидное
 */
export default function maxDateFromField(
    dateOffset: IDateOffsetFromField,
    value?: DateLikeType,
    formData?: IFormValidationData,
): boolean {
    if (value && formData) {
        const {field, strict, comparisonScale} = dateOffset;
        const {formValues, groupPath} = formData;
        let group = field.fieldGroupId
            ? _get(formValues, field.fieldGroupId)
            : formValues;

        if (Array.isArray(group)) {
            group = _get(formValues, groupPath);
        }

        if (group) {
            const maxDate = getDateWithOffset(
                group[field.fieldName],
                dateOffset,
                HUMAN_DATE_RU,
            );
            const dateValue = moment(value, HUMAN_DATE_RU);

            if (maxDate.isValid()) {
                return strict
                    ? dateValue.isBefore(maxDate, comparisonScale)
                    : dateValue.isSameOrBefore(maxDate, comparisonScale);
            }
        }
    }

    return true;
}
