import fse from 'fs-extra';
import path from 'path';
import mapValues from 'lodash/mapValues';
import pickBy from 'lodash/pickBy';
import isEmpty from 'lodash/isEmpty';

import getTmpDirPath from './utilities/getTmpDirPath';
import getTankerUrl from './utilities/getTankerUrl';
import renderKeySetListHtml from './utilities/renderKeySetListHtml';

const OUTPUT = path.resolve(getTmpDirPath(), 'delta-tanker-report.html');

const prevUsedKeysPath = path.resolve(
    getTmpDirPath(),
    'used-tanker-keys.previous.json',
);

const usedKeysPath = path.resolve(getTmpDirPath(), 'used-tanker-keys.json');

(async (): Promise<void> => {
    let prevUsedKeys;

    try {
        prevUsedKeys = getUsed(prevUsedKeysPath);
    } catch (e) {
        prevUsedKeys = {};

        console.warn('Previous usedjson ');
    }

    const usedKeys = getUsed(usedKeysPath);

    const diff = mapValues(prevUsedKeys, (keySet, keySetName) => {
        const removedKeys = pickBy(
            keySet,
            (_, key) => !usedKeys?.[keySetName]?.[key],
        );

        return mapValues(removedKeys, (_, key) =>
            getTankerUrl(keySetName, key),
        );
    });

    const diffWithoutEmpty = pickBy(diff, keySet => !isEmpty(keySet));

    await fse.ensureDir(getTmpDirPath());

    await fse.writeFile(
        OUTPUT,
        renderKeySetListHtml(diffWithoutEmpty, [], {
            title: 'Tanker removed keys report',
            description:
                'Ключи которые больше не используются в коде (diff используемых ключей между билдами ветки)',
        }),
    );
})();

function getUsed(filePath: string): Record<string, Record<string, true>> {
    try {
        return require(filePath);
    } catch (e) {
        if (e instanceof Error) {
            throw Error(`Used keys json ${filePath} not loaded: ${e.message}`);
        }

        throw e;
    }
}
