import traverse, {Node} from '@babel/traverse';
import {set, last} from 'lodash';

import * as t from '@babel/types';

export interface IFindUsagesResult {
    /**
     * Точное использование ключа (трансформированного) и кейсета
     */
    preciseUsages: Record<string, Record<string, true>>;
    /**
     * Хитрое использование кейсетов
     */
    unsafeUsages: Record<string, true>;
}

/**
 * Находит использование кейсетов
 */
export default function findUsages(ast: Node): IFindUsagesResult {
    const result: IFindUsagesResult = {
        preciseUsages: {},
        unsafeUsages: {},
    };

    traverse(ast, {
        enter(path) {
            const {node: importDeclaration, scope} = path;

            if (!t.isImportDeclaration(importDeclaration)) {
                return;
            }

            const source = importDeclaration.source;

            if (
                !t.isStringLiteral(source) ||
                !source.value.startsWith('i18n/')
            ) {
                return;
            }

            const keySetName = last(source.value.split('/'));

            if (!keySetName) {
                throw new Error('keySetName is undefined');
            }

            const specifiers = importDeclaration.specifiers;

            specifiers.forEach(s => {
                if (t.isImportSpecifier(s) && t.isIdentifier(s.imported)) {
                    set(
                        result.preciseUsages,
                        [keySetName, s.imported.name],
                        true,
                    );

                    return;
                }

                if (t.isImportNamespaceSpecifier(s)) {
                    const localVariableName = s.local.name;

                    traverse(scope.block, {
                        enter(identifierPath) {
                            const identifier = identifierPath.node;

                            if (
                                !t.isIdentifier(identifier) ||
                                identifier.name !== localVariableName
                            ) {
                                return;
                            }

                            const identifierParent = identifierPath.parent;

                            if (
                                t.isImportNamespaceSpecifier(identifierParent)
                            ) {
                                return;
                            }

                            if (
                                !t.isMemberExpression(identifierParent) ||
                                identifierParent.computed ||
                                !t.isIdentifier(identifierParent.property)
                            ) {
                                set(result.unsafeUsages, [keySetName], true);

                                return;
                            }

                            set(
                                result.preciseUsages,
                                [keySetName, identifierParent.property.name],
                                true,
                            );
                        },
                    });

                    return;
                }

                throw new Error('Unexpected Import specifier');
            });
        },
    });

    return result;
}
