const path = require('path');
const webpack = require('webpack');
const {mergeWithRules, CustomizeRule} = require('webpack-merge');
const MiniCSSExtractPlugin = require('mini-css-extract-plugin');
const TerserPlugin = require('terser-webpack-plugin');
const CssMinimizerPlugin = require('css-minimizer-webpack-plugin');
const StatsWebpackWriterPlugin = require('stats-webpack-plugin');
const {BundleAnalyzerPlugin} = require('webpack-bundle-analyzer');
const LoadablePlugin = require('@loadable/webpack-plugin');
const {RetryChunkLoadPlugin} = require('webpack-retry-chunk-load-plugin');
const CopyPlugin = require('copy-webpack-plugin');

const {EWebpackTarget} = require('types/EWebpackTarget');

const uglifyJsPluginSettings = require('./utilities/getUglifyJsPluginSettings');
const getFileNameByBuildType = require('./utilities/getFileNameByBuildType');
const {
    srcPath,
    htmlInputPath,
    clientBuildPath,
    serverBuildPath,
    excludeNodeModulesRegExpPath,
    includeYandexMessengerRegExpPath,
    staticPath,
} = require('./utilities/getPathParams');
const getCssModulesLocalIdentName = require('./utilities/getCssModulesLocalIdentName');
const getStaticFileNameSuffix = require('./utilities/getStaticFileNameSuffix');

const {
    isDevelopment,
    isProduction,
    buildAnalyzerMode,
    isBuildBundleAnalyzer,
    withHotReload,
    isStorybookBuild,
} = require('./variables');
const createCommonWebpackConfig = require('./common.webpack.config');
const {getJsRules} = require('./getJsRules');

const commonWebpackConfig = createCommonWebpackConfig({isServer: false});
const staticFileNameSuffix = getStaticFileNameSuffix();

const clientEntry = [path.join(srcPath, 'client.tsx')];
const clientAmpEntry = [
    path.join(srcPath, 'containers', 'AppAmp', 'AppAmp.tsx'),
];
const clientMinimizerPlugins = [];
const clientWebpackPlugins = [
    new RetryChunkLoadPlugin(),
    new MiniCSSExtractPlugin({
        filename: getFileNameByBuildType({
            fileType: 'css',
            isProduction,
            suffix: staticFileNameSuffix,
        }),
        ignoreOrder: true,
    }),
    new webpack.NormalModuleReplacementPlugin(
        /server\/redux\/.+?/,
        'lodash/noop',
    ),
    new CopyPlugin({
        patterns: [
            {
                from: path.join(htmlInputPath, '*'),
                to: `${clientBuildPath}/[name][ext]`,
            },
        ],
    }),
];

if (!isStorybookBuild) {
    clientWebpackPlugins.push(
        new LoadablePlugin({
            writeToDisk: {
                filename: serverBuildPath,
            },
        }),
    );
}

const rumExternalScripts = [
    'implementation.min.js',
    'send.min.js',
    'onload.min.js',
    'retries.min.js',
].map(bundleName => `@yandex-int/rum-counter/dist/bundle/${bundleName}`);

if (isDevelopment && withHotReload) {
    clientEntry.unshift('webpack-hot-middleware/client');
    clientWebpackPlugins.unshift(new webpack.HotModuleReplacementPlugin());
}

if (isProduction) {
    clientMinimizerPlugins.push(
        new TerserPlugin(uglifyJsPluginSettings),
        new CssMinimizerPlugin({
            minimizerOptions: {
                preset: [
                    'default',
                    {
                        discardComments: {removeAll: true},
                    },
                ],
            },
        }),
    );
    clientWebpackPlugins.push(
        new StatsWebpackWriterPlugin('../stats.json', {
            all: false,
            assets: true,
            publicPath: true,
            warningsFilter: () => true,
        }),
    );
}

if (isBuildBundleAnalyzer) {
    clientWebpackPlugins.push(
        new BundleAnalyzerPlugin({
            analyzerMode: buildAnalyzerMode,
            openAnalyzer: false,
            reportFilename:
                buildAnalyzerMode === 'json'
                    ? 'bundle-stats.json'
                    : 'stats.html',
        }),
    );
}

/* Styles Loaders */
const miniCssExtractLoader = MiniCSSExtractPlugin.loader;
const prependScssLoaders = [
    miniCssExtractLoader,
    {
        loader: 'css-loader',
        options: {
            importLoaders: 4,
            modules: {
                localIdentName: getCssModulesLocalIdentName(isProduction),
            },
        },
    },
];
const prependCssLoaders = [miniCssExtractLoader, 'css-loader'];

if (isDevelopment && withHotReload) {
    prependScssLoaders.splice(1, 0, 'css-hot-loader');
}

export const clientWebpackConfig = {
    name: 'client',
    devtool: isDevelopment ? 'cheap-module-source-map' : 'hidden-source-map',
    target: 'browserslist',
    entry: {
        client: clientEntry,
        clientAmp: clientAmpEntry,
        rum: rumExternalScripts,
    },
    devServer: {
        static: {
            directory: 'build/client',
        },
    },
    output: {
        path: clientBuildPath,
        publicPath: staticPath,
        filename: getFileNameByBuildType({
            fileType: 'js',
            isProduction,
            suffix: staticFileNameSuffix,
        }),
        chunkFilename: getFileNameByBuildType({
            fileType: 'js',
            isProduction,
            suffix: staticFileNameSuffix,
        }),
        crossOriginLoading: 'anonymous',
    },
    module: {
        rules: [
            ...getJsRules(EWebpackTarget.WEB),

            {
                test: /\.scss$/,
                exclude: excludeNodeModulesRegExpPath,
                use: prependScssLoaders,
                resolve: {
                    extensions: ['.scss', '.sass'],
                },
            },
            {
                test: /\.css$/,
                include: includeYandexMessengerRegExpPath,
                use: prependCssLoaders,
            },
            {
                test: /\.css$/,
                exclude: excludeNodeModulesRegExpPath,
                use: prependCssLoaders,
            },
        ],
    },
    plugins: clientWebpackPlugins,
    optimization: {
        minimizer: clientMinimizerPlugins,
        splitChunks: {
            chunks: 'all',
        },
        realContentHash: false,
        runtimeChunk: 'single',
    },
    cache: isDevelopment
        ? {
              type: 'filesystem',
          }
        : false,
};

module.exports = mergeWithRules({
    module: {
        rules: {
            test: CustomizeRule.Match,
            include: CustomizeRule.Match,
            exclude: CustomizeRule.Match,
            use: CustomizeRule.Prepend,
        },
    },
})(commonWebpackConfig, clientWebpackConfig);
