const path = require('path');
const webpack = require('webpack');
const {mergeWithRules, CustomizeRule} = require('webpack-merge');
const nodeExternals = require('webpack-node-externals');
const TerserPlugin = require('terser-webpack-plugin');

const {EWebpackTarget} = require('types/EWebpackTarget');

const {
    rootPath,
    serverBuildPath,
    excludeNodeModulesRegExpPath,
} = require('./utilities/getPathParams');
const getCssModulesLocalIdentName = require('./utilities/getCssModulesLocalIdentName');

const {
    isDevelopment,
    isProduction,
    serverSourceMap,
    buildParallelLimit,
} = require('./variables');
const createCommonWebpackConfig = require('./common.webpack.config');
const {getJsRules} = require('./getJsRules');

const commonWebpackConfig = createCommonWebpackConfig({isServer: true});

const renderServerEntry = [path.join(rootPath, 'server', 'renderServer.js')];
const renderAmpServerEntry = [
    path.join(rootPath, 'server', 'renderAmpServer.js'),
];
const apiControllerEntry = [path.join(rootPath, 'server', 'apiController.js')];

const serverWebpackPlugins = [
    new webpack.NormalModuleReplacementPlugin(/sagas/, 'lodash/noop'),
];

/* Styles loaders */
const prependScssLoaders = [
    {
        loader: 'css-loader',
        options: {
            url: false,
            import: false,
            modules: {
                localIdentName: getCssModulesLocalIdentName(isProduction),
                exportOnlyLocals: true,
            },
        },
    },
];

const prependCssLoaders = [
    {
        loader: 'css-loader',
        options: {
            url: false,
            import: false,
            modules: {
                exportOnlyLocals: true,
            },
        },
    },
];

const serverWebpackConfig = {
    name: 'server',
    target: 'node12.22',
    devtool: serverSourceMap,
    externals: [nodeExternals({allowlist: [/@yandex-lego/]})],
    entry: {
        renderServer: renderServerEntry,
        renderAmpServer: renderAmpServerEntry,
        apiController: apiControllerEntry,
    },
    output: {
        path: serverBuildPath,
        libraryTarget: 'commonjs2',
    },
    module: {
        rules: [
            ...getJsRules(EWebpackTarget.NODE),

            {
                test: /\.scss$/,
                exclude: excludeNodeModulesRegExpPath,
                use: prependScssLoaders,
            },
            {
                test: /\.css$/,
                exclude: excludeNodeModulesRegExpPath,
                use: prependCssLoaders,
            },
        ],
    },
    plugins: serverWebpackPlugins,
    optimization: {
        minimizer: [
            new TerserPlugin({
                terserOptions: {
                    keep_classnames: true,
                    keep_fnames: true,
                },
                parallel: buildParallelLimit,
            }),
        ],
        realContentHash: false,
    },
    cache: isDevelopment
        ? {
              type: 'filesystem',
          }
        : false,
};

module.exports = mergeWithRules({
    module: {
        rules: {
            test: CustomizeRule.Match,
            include: CustomizeRule.Match,
            exclude: CustomizeRule.Match,
            use: CustomizeRule.Prepend,
        },
    },
})(commonWebpackConfig, serverWebpackConfig);
