/* eslint no-unused-vars: 0 */

import isObject from 'lodash/isObject';
import mapValues from 'lodash/mapValues';
import 'location-origin';
import 'classlist-polyfill';
import 'intersection-observer';
import dataset from 'element-dataset';
import cookies from 'cookies-js';
import page from 'page';
import React from 'react';
import {hydrate} from 'react-dom';
import {Provider} from 'react-redux';

dataset();

import Platform from '../common/interfaces/Platform';

import {Root, Loadable, moment} from '../common/components/Root/Root';
import logger from './logger';

import routerAdapter from './routerAdapter';
import {api, ApiError} from './api';
import bindIntervals from './bindIntervals';
import setBeforeUnload from './setBeforeUnload';

import searchAnalysis from './listeners/searchAnalysis';
import setupPricesPolling from './listeners/setupPricesPolling';
import setupBlablacarPolling from './listeners/setupBlablacarPolling';
import setupThreadPricePolling from './listeners/setupThreadPricePolling';
import setupThreadBlablacarPolling from './listeners/setupThreadBlablacarPolling';
import {
    initSearchHistory,
    setupSearchHistory,
} from './listeners/setupSearchHistory';

import {decodeSpecialCharacters} from '../common/lib/stringUtils';
import router from '../common/router';
import injectToCreateStore from '../common/store';
import {params, onMetrikaInit, getClientId} from '../common/lib/yaMetrika';
import setupTimeCorrection from './lib/setupTimeCorrection';
import {
    yandexSansDownloadObserver,
    YANDEX_SANS_COOKIE,
} from './helpers/yandexSansDownloadObserver';
import domContentLoaded from './helpers/domContentLoaded';

import {setClientEnvironment} from '../common/actions/environment';
import {setClientId} from '../common/actions/user';

import StaticContext from '../common/components/StaicContext/StaticContext';
import PageErrorBoundary from '../common/components/PageErrorBoundary/PageErrorBoundary';

const isMobile = process.env.PLATFORM === Platform.mobile;

const stateReviver = val => {
    if (typeof val === 'string') {
        return decodeSpecialCharacters(val);
    }

    if (Array.isArray(val)) {
        return val.map(item => stateReviver(item));
    }

    if (isObject(val)) {
        return mapValues(val, stateReviver);
    }

    return val;
};

window.INITIAL_STATE = mapValues(window.INITIAL_STATE, stateReviver);
window.STATIC_CONTEXT = mapValues(window.STATIC_CONTEXT, stateReviver);

const injections = {api, cookies, logger, ApiError};
const store = injectToCreateStore(injections)(window.INITIAL_STATE);

const state = store.getState();

if (!state.environment.production) {
    // подменяем XMLHttpRequest и fetch
    const open = window.XMLHttpRequest.prototype.open;
    const realFetch = window.fetch;
    const replaceTargetRef = function (url) {
        // подменяем запросы к директу для того, чтобы видеть рекламу
        const match =
            Boolean(url) &&
            url.match(
                /^https:\/\/an\.yandex\.ru\/(.*)target-ref=([^&]*)(&.*|$)/,
            );

        if (match) {
            const newTargetRef = match[2].replace(
                /^(https?%3A%2F%2F)[.\w-]*yandex(?:-team)?\.(ru|ua|by|kz|uz)[%\w]*?%2F/i,
                `https%3A%2F%2F${isMobile ? 't.' : ''}rasp.yandex.$2%2F`,
            );

            return `https://an.yandex.ru/${match[1]}target-ref=${newTargetRef}${match[3]}`;
        }

        return url;
    };

    window.XMLHttpRequest.prototype.open = function (...args) {
        args[1] = replaceTargetRef(args[1]);

        return open.bind(this)(...args);
    };

    window.fetch = function (...args) {
        args[0] = replaceTargetRef(args[0]);

        return realFetch.bind(this)(...args);
    };
}

if (state.linkToTracer) {
    // eslint-disable-next-line no-console
    console.log(state.linkToTracer);
}

const adapter = routerAdapter(page, store, injections);

moment.locale(store.getState().language);

const timeCorrectionPromise = setupTimeCorrection(api);

let hydratePromiseResolve;

const hydratePromise = new Promise(resolve => {
    hydratePromiseResolve = resolve;
});

onMetrikaInit(async () => {
    params(state.flags);
    await hydratePromise;
    store.dispatch(setClientId(getClientId()));
}, state.isTouch);

(async () => {
    await domContentLoaded();
    await Promise.all([timeCorrectionPromise, Loadable.preloadReady()]);

    const root = document.getElementById('root');

    if (state.flags.__showFlagsInConsole) {
        // eslint-disable-next-line no-console
        console.log(state.flags);
    }

    hydrate(
        <StaticContext.Provider value={window.STATIC_CONTEXT}>
            <Provider store={store}>
                <PageErrorBoundary>
                    <Root />
                </PageErrorBoundary>
            </Provider>
        </StaticContext.Provider>,
        root,
    );

    hydratePromiseResolve();
    store.dispatch(setClientEnvironment());
    initSearchHistory(store);
    searchAnalysis(store, api);
    setupPricesPolling(store, api);
    setupThreadPricePolling(store, api);
    setupThreadBlablacarPolling(store, api);
    setupBlablacarPolling(store, api);
    setupSearchHistory(store);

    // Проверяем куки на признак загруженных шрифтов
    if (!cookies.get(YANDEX_SANS_COOKIE)) {
        yandexSansDownloadObserver(store);
    }

    page();
    router(adapter, {isServer: false});
    bindIntervals(store);
    setBeforeUnload(store);
})();
