import get from 'lodash/get';
import moment from 'moment';

import {TRAIN_TYPE} from '../../common/lib/transportType';
import {MOMENT_WITHOUT_TIMEZONE} from '../../common/lib/date/formats';

import {isSearch} from '../../common/lib/page';
import isAllDaysSearch from '../../common/lib/search/isAllDaysSearch';
import observeStore from '../lib/observeStore';
import {prepareBusTariffsData} from '../../common/lib/logging';
import areAllSegmentsWithPrice from '../../common/lib/segments/areAllSegmentsWithPrice';
import getDirectSegments from '../../common/lib/segments/getDirectSegments';

import {params} from '../../common/lib/yaMetrika';
import reachGoalYBus from '../../common/lib/metrika/reachGoalYBus';
import reachGoalTrainsRatio from '../../common/lib/metrika/reachGoalTrainsRatio';
import reachGoalSegmentsRatio from '../../common/lib/metrika/reachGoalSegmentsRatio';
import reachGoalsTransportTypes from '../../common/lib/metrika/reachGoalsTransportTypes';
import reachGoalTomorrowSegments from '../../common/lib/metrika/reachGoalTomorrowSegments';
import getBadgesData from '../../common/lib/segments/badges/getBadgesData';
import getAppropriateBadges from '../../common/lib/segments/badges/getAppropriateBadges';
import {setBadges} from '../../common/actions/search';

export default function searchAnalysis(store, api) {
    function searchCompleted(state) {
        const {search, page} = state;

        return page.fetching === null && !search.queryingPrices;
    }

    function reachGoals() {
        const {page, search} = store.getState();
        const {context, segments} = search;

        reachGoalTomorrowSegments(segments, context);
        reachGoalSegmentsRatio(segments, page, context);
        reachGoalsTransportTypes(segments, context);
        reachGoalTrainsRatio(segments);
        reachGoalYBus(segments, context);
    }

    function logBusTariffs() {
        const {search, nationalVersion} = store.getState();
        const data = prepareBusTariffsData(search);

        if (data) {
            api.exec('busTariffsLogger', {
                data,
                nationalVersion,
                context: search.context,
            });
        }
    }

    // для поездов отправляем в параметры визита метрики количество разных дат старта продаж билетов
    function logDifferentStartSalesTimes() {
        const {getState} = store;
        const {search, language} = getState();
        const {context: searchContext, segments} = search;

        const directSegments = getDirectSegments(segments, TRAIN_TYPE);

        const dontRequestPartnerSchedule =
            searchContext.searchForPastDate ||
            !directSegments.length ||
            isAllDaysSearch(searchContext) ||
            areAllSegmentsWithPrice(directSegments);

        if (dontRequestPartnerSchedule) {
            return;
        }

        api.exec('partnerSchedule', {
            pointFrom: searchContext.from.key,
            pointTo: searchContext.to.key,
            when: moment(searchContext.when.date).format(
                MOMENT_WITHOUT_TIMEZONE,
            ),
            language,
        }).then(data => {
            const startSalesDateTimesCount = get(
                data,
                'schedule.startSalesDateTimes.length',
            );

            if (!startSalesDateTimesCount) {
                return;
            }

            params({
                startSalesDateTimes: startSalesDateTimesCount,
            });
        });
    }

    /**
     * Выставляем сегментам бейджики (самый дешевый, самый быстрый и тд)
     */
    function badgesDataHandler() {
        const {getState, dispatch} = store;
        const {search} = getState();
        const {segments} = search;

        if (!segments.length) {
            return;
        }

        const badgesData = getBadgesData(segments);

        if (!badgesData) {
            return;
        }

        const segmentsWithBadges = segments.map(segment => ({
            ...segment,
            badges: getAppropriateBadges(segment, badgesData),
        }));

        dispatch(setBadges(segmentsWithBadges));
    }

    function observeHandler() {
        const state = store.getState();

        if (isSearch(state) && searchCompleted(state)) {
            reachGoals();
            logBusTariffs();
            logDifferentStartSalesTimes();
            badgesDataHandler();
        }
    }

    observeStore(store, observeHandler, 'page', 'fetching');

    observeStore(store, observeHandler, 'search', 'queryingPrices');
}
