import DateSpecialValue from '../../common/interfaces/date/DateSpecialValue';

import logger from '../logger';
import {setBlablacar} from '../../common/actions/search';
import {isSearch} from '../../common/lib/page';
import observeStore from '../lib/observeStore';
import {getIncrementalTimeoutPromise} from '../../common/lib/timeoutPromise';
import {
    InterruptError,
    PollingInterruptError,
} from '../../common/lib/errors/executionErrors';
import createSearchContextChecker from '../../common/lib/createSearchContextChecker';
import isRidesharingAvailableForRequest from '../../common/lib/segments/ridesharing/isRidesharingAvailableForRequest';
import {ApiError} from '../api';

const incrementalTimeout = getIncrementalTimeoutPromise(1000, 4000);

export default function setupBlablacarPolling(store, api) {
    const {getState, dispatch} = store;

    let allDaysPolling = false;
    let pollingInProcess = false;

    function availableForPollingOnAllDays() {
        const {search} = getState();
        const {context, blablacar} = search;
        const {tariff, banned} = blablacar;

        return (
            !allDaysPolling &&
            !banned &&
            context.when.special !== 'all-days' &&
            Boolean(tariff) &&
            tariff.offersCount === 0
        );
    }

    function getParams(poll = true) {
        const {search, nationalVersion, isTouch, flags} = getState();
        const {context} = search;
        const {from, to, when} = context;

        return {
            pointFrom: from.key,
            pointTo: to.key,
            nationalVersion,
            ...(allDaysPolling ? null : {date: when.date}),
            poll,
            isMobile: isTouch,
            flags,
        };
    }

    function dispatchBlablacar({blablacar}) {
        const {
            isTouch,
            search: {context},
        } = getState();
        const {querying} = blablacar;

        return dispatch(
            setBlablacar({
                blablacar: {
                    ...(allDaysPolling
                        ? getState().search.blablacar
                        : blablacar),
                    querying,
                    allDaysCheckComplete: allDaysPolling
                        ? querying === false
                        : context.when.special === DateSpecialValue.allDays,
                    allDaysCheckResult: allDaysPolling
                        ? blablacar.tariff
                        : null,
                },
                isMobile: isTouch,
            }),
        );
    }

    function requestBlablacar() {
        const {blablacar} = getState().search;

        return blablacar.querying
            ? api.exec('blablacar3', getParams())
            : new Promise(resolve => resolve({blablacar}));
    }

    function checkQuerying(value) {
        const {blablacar} = getState().search;

        if (blablacar.querying) {
            return value;
        }

        if (availableForPollingOnAllDays()) {
            allDaysPolling = true;

            return api
                .exec('blablacar3', getParams(false))
                .then(dispatchBlablacar)
                .then(checkQuerying);
        }

        return Promise.reject(new PollingInterruptError('querying-end'));
    }

    function pollBlablacar(checkContext, retries) {
        const fetchPrices = () => {
            retries++;

            return requestBlablacar().catch(err => {
                if (err instanceof ApiError) {
                    return incrementalTimeout(retries)
                        .then(checkContext)
                        .then(fetchPrices);
                }

                return err;
            });
        };

        return fetchPrices()
            .then(checkContext) // проверяем актуальность контекста (пользователь не запустил другой поиск)
            .then(dispatchBlablacar) // устанавливаем полученные цены
            .then(checkQuerying) // проверяем, нужно ли запрашивать еще цены
            .then(() => incrementalTimeout(retries)) // ждём
            .then(checkContext) // проверяем что контекст всё еще актуальный
            .then(() => pollBlablacar(checkContext, retries)) // запрашиваем по новому кругу
            .catch(e => {
                pollingInProcess = false;
                allDaysPolling = false;

                if (!(e instanceof InterruptError)) {
                    logger.error('blablacarPolling', e);
                }
            });
    }

    function onBlablacarChange(blablacar) {
        const state = getState();
        const {
            search: {
                context: {searchForPastDate, from, to},
            },
            flags,
        } = state;

        const bbkIsAvailableForRequest = isRidesharingAvailableForRequest({
            searchForPastDate,
            from,
            to,
            flags,
        });

        if (
            isSearch(state, true) &&
            bbkIsAvailableForRequest &&
            !pollingInProcess &&
            blablacar
        ) {
            pollingInProcess = true;
            pollBlablacar(createSearchContextChecker(getState), 0);
        }
    }

    return observeStore(store, onBlablacarChange, 'search', 'blablacar');
}
