import {ApiError} from '../api';
import {FilterTransportType} from '../../common/lib/transportType';
import logger from '../logger';
import observeStore from '../lib/observeStore';
import {isSearch} from '../../common/lib/page';
import {getIncrementalTimeoutPromise} from '../../common/lib/timeoutPromise';
import isAllDaysSearch from '../../common/lib/search/isAllDaysSearch';
import {
    InterruptError,
    PollingInterruptError,
} from '../../common/lib/errors/executionErrors';
import createSearchContextChecker from '../../common/lib/createSearchContextChecker';
import joinDynamicPriceAnswers from '../../common/lib/search/joinDynamicPriceAnswers';
import getTimeRangeForTrainTariffs from '../../common/lib/segments/tariffs/getTimeRangeForTrainTariffs';
import joinTransferAnswers from '../../common/lib/search/joinTransferAnswers';
import requestTransfer from '../../common/lib/search/requestTransfer';

import {updateFilters} from '../../common/actions/search/filters';
import {
    updatePrices,
    setQuerying,
    setPricesAreCached,
    setPlaneQueries,
    upsertTransferSegments,
    setTransferQuerying,
    setNeedInitialRequestForTrainTariffs,
} from '../../common/actions/search';

const timeout = getIncrementalTimeoutPromise(1000, 4000);

export default function setupPricesPolling(store, api) {
    const {getState, dispatch} = store;
    const {currencies} = getState();

    function planePriceIsQuerying() {
        return getState().search.querying.plane;
    }

    function trainPriceIsQuerying() {
        return getState().search.querying.train;
    }

    function transferAllIsQuerying() {
        return getState().search.querying.transferAll;
    }

    function transferTrainIsQuerying() {
        return getState().search.querying.transferTrain;
    }

    function transferPlaneIsQuerying() {
        return getState().search.querying.transferPlane;
    }

    function busPriceIsQuerying() {
        return getState().search.querying.bus;
    }

    /**
     * Сшивает вместе ответы от разных qid для авиа
     * @param {Array} answers
     * @param {Object} search
     * @return {{fetchedAnswers: {Array}, planeQueries: {Array}}}
     */
    function fetchPlaneAnswers(answers, search) {
        const {planeQueries: currentPlaneQueries} = search;
        const newPlaneQueries = [];
        const fetchedPlaneAnswers = answers.reduce(
            (result, answer) => {
                if (answer.planeTariffs) {
                    const {planeTariffs} = answer;
                    const {segments, qid, querying} = planeTariffs;
                    const currentPlaneQuery = currentPlaneQueries.filter(
                        query => query.id === qid,
                    )[0];

                    if (!currentPlaneQuery) {
                        return result;
                    }

                    const planeQuery = {
                        id: qid,
                        skip_partners: new Set(currentPlaneQuery.skip_partners),
                    };

                    if (segments) {
                        result.segments.push(...segments);
                        segments.forEach(segment => {
                            if (segment.tariffs && segment.tariffs.partner) {
                                planeQuery.skip_partners.add(
                                    segment.tariffs.partner,
                                );
                            }
                        });
                    }

                    // Превращаем сет в массив для записи в стор
                    planeQuery.skip_partners = [...planeQuery.skip_partners];

                    if (querying) {
                        result.querying = querying;
                        newPlaneQueries.push(planeQuery);
                    }
                }

                return result;
            },
            {segments: [], querying: false},
        );
        const fetchedAnswers = answers.filter(answer => !answer.planeTariffs);

        fetchedAnswers.push({planeTariffs: fetchedPlaneAnswers});

        return {
            fetchedAnswers,
            planeQueries: newPlaneQueries,
        };
    }

    function setPrices(answers) {
        const query = [];
        const {transfers, status} = joinTransferAnswers(answers);

        const {fetchedAnswers, planeQueries} = fetchPlaneAnswers(
            answers,
            getState().search,
        );
        const {segments, querying} = joinDynamicPriceAnswers(
            fetchedAnswers,
            currencies,
        );

        // устанавливаем тарифы сегментов
        if (segments.length) {
            query.push(dispatch(updatePrices({segments}, getState())));
        }

        // устанавливаем тарифы пересадок
        if (transfers.length) {
            query.push(dispatch(upsertTransferSegments(transfers, getState())));
        }

        dispatch(setPlaneQueries({planeQueries}, getState()));

        // снимаем флаг кеширования цен, т.к. на клиенте опрос происходит без кеширования
        dispatch(setPricesAreCached(false));

        // выставляем флаги опроса цен (plane, train, bus)
        query.push(dispatch(setQuerying(querying, getState())));

        // выставляем флаги опроса цен пересадок
        if (Object.keys(status).length) {
            query.push(dispatch(setTransferQuerying(status)));
        }

        // Если пришли новые сегменты и опрос цен закончен - обновляем фильтры
        if (segments.length || !getState().search.queryingPrices) {
            query.push(dispatch(updateFilters()));
        }

        return Promise.all(query);
    }

    /**
     * запрос цен, если для какой-то ручки опрос цен закончился - её не дёргаем
     * @param {boolean} poll
     * @return {Promise}
     */
    function requestPrices(poll = true) {
        const {
            flags,
            environment: {type: environmentType},
            user: {isBot},
            search,
            tld,
            platform,
            nationalVersion,
        } = getState();
        const {context, planeQueries, needInitialRequestForTrainTariffs} =
            search;
        const {
            transportType,
            language,
            from: {key: pointFrom},
            to: {key: pointTo},
            when: {date: when},
            time: {now},
        } = context;

        const query = [];

        if (planePriceIsQuerying()) {
            planeQueries.forEach(planeQuery => {
                query.push(
                    api.exec('planeTariffs', {
                        qid: planeQuery.id,
                        skip_partners: planeQuery.skip_partners,
                        language: context.language,
                    }),
                );
            });
        }

        if (trainPriceIsQuerying()) {
            const pollTrainTariffs = needInitialRequestForTrainTariffs
                ? false
                : poll;

            if (needInitialRequestForTrainTariffs) {
                dispatch(setNeedInitialRequestForTrainTariffs(false));
            }

            const timeRange = getTimeRangeForTrainTariffs(search.segments);

            if (timeRange) {
                query.push(
                    api.exec('trainTariffs2', {
                        transportType,
                        pointFrom,
                        pointTo,
                        now,
                        language,
                        nationalVersion,
                        flags,
                        poll: pollTrainTariffs,
                        environmentType,
                        ...timeRange,
                    }),
                );
            }
        }

        if (busPriceIsQuerying()) {
            query.push(
                api.execBusTariffs2({
                    flags,
                    context,
                    nationalVersion,
                    environmentType,
                }),
            );
        }

        const requestTransferParams = {
            pointFrom,
            pointTo,
            when,
            language,
            tld,
            poll,
            api,
            flags,
            platform,
            isBot,
        };

        if (transferAllIsQuerying()) {
            query.push(
                requestTransfer({
                    ...requestTransferParams,
                    transportType: FilterTransportType.all,
                }),
            );
        }

        if (transferTrainIsQuerying()) {
            query.push(
                requestTransfer({
                    ...requestTransferParams,
                    transportType: FilterTransportType.train,
                }),
            );
        }

        if (transferPlaneIsQuerying()) {
            query.push(
                requestTransfer({
                    ...requestTransferParams,
                    transportType: FilterTransportType.plane,
                }),
            );
        }

        return Promise.all(query);
    }

    function checkQuerying(value) {
        // Если хотя бы для одной из ручек опрос не закончился - продолжаем опрашивать
        if (
            planePriceIsQuerying() ||
            trainPriceIsQuerying() ||
            busPriceIsQuerying() ||
            transferAllIsQuerying() ||
            transferTrainIsQuerying() ||
            transferPlaneIsQuerying()
        ) {
            return value;
        }

        return Promise.reject(new PollingInterruptError('querying-end'));
    }

    function pollPrices(checkContext, poll, retries) {
        const fetchPrices = () => {
            retries++;

            return requestPrices(poll).catch(err => {
                if (err instanceof ApiError) {
                    logger.error('client/listeners/setupPricesPolling', err);

                    return timeout(retries)
                        .then(checkContext)
                        .then(fetchPrices);
                }

                return err;
            });
        };

        return (
            fetchPrices()
                // проверяем актуальность контекста (пользователь не запустил другой поиск)
                .then(checkContext)

                // устанавливаем полученные цены
                .then(setPrices)

                // проверяем, нужно ли запрашивать еще цены
                .then(checkQuerying)

                // ждём 2^кол-во попыток секунд
                .then(() => timeout(retries))

                // проверяем что контекст всё еще актуальный
                .then(checkContext)

                // запрашиваем по новому кругу
                .then(() => pollPrices(checkContext, true, retries))

                .catch(e => {
                    if (!(e instanceof InterruptError)) {
                        logger.error('pricePolling', e);
                    }
                })
        );
    }

    function onPricesQueryingChange(queryingPrices) {
        const state = getState();
        const {context, pricesAreCached} = state.search;

        // при поиске на все дни получаем цены из логов - опрос не требуется
        // также опрос цен не требуется, если нам пришли архивные данные
        if (
            isSearch(state, true) &&
            !isAllDaysSearch(context) &&
            queryingPrices
        ) {
            const checkContext = createSearchContextChecker(getState);

            pollPrices(checkContext, !pricesAreCached, 0);
        }
    }

    const teardownPricesPolling = observeStore(
        store,
        onPricesQueryingChange,
        'search',
        'queryingPrices',
    );

    return teardownPricesPolling;
}
