import {parse} from 'qs';

import logger from './logger';
import noop from '../common/lib/noop';
import Error404 from '../common/lib/errors/Error404';

/**
 * Функция redirect для response
 * поодерживаем количество аргументов как у redirect express'a
 * @param {Function} page - библиотека для роутинга
 * @param {Array} args - массив аргументов
 * @param {string} args.0 - если args.length === 2, тогда статус для редиректа, иначе урл
 * @param {string}[args.1] - урл для редиректа
 */
function resRedirect(page, ...args) {
    const url = args.length === 2 ? args[1] : args[0];

    // если это внешняя ссылка, то просто меняем location
    if (/^http/.test(url)) {
        window.location = url;
    } else {
        page.redirect(url);
    }
}

// обработка ошибок роутера
function nextWrapper(next) {
    return err => {
        if (err && err instanceof Error404) {
            return next();
        }

        if (err) {
            logger.error('client/routerAdapter', err);
        }

        return next(err);
    };
}

export default function routerAdapter(page, store, injections) {
    page('*', (req, next) => {
        req.query = parse(req.querystring);
        next();
    });

    const wrap = cb => (req, next) => {
        try {
            req.originalUrl = req.path;

            return cb({
                ...injections,
                store,
                next: nextWrapper(next),
                req,
                res: {
                    redirect: resRedirect.bind(null, page),
                    render: () => Promise.resolve(),
                    status: noop,
                },
            });
        } catch (err) {
            return next(err);
        }
    };

    const match = (route, cb) => page(route, wrap(cb));

    return {
        get: match,
        post: match,
        use: match,
    };
}
