import {batch} from 'react-redux';
import {createAction} from 'redux-actions';

import {encodeSpecialCharacters} from '../lib/stringUtils';

export const START_FETCHING_PAGE = 'START_FETCHING_PAGE';
export const startFetchingPage = createAction(START_FETCHING_PAGE);

export const FINISH_FETCHING_PAGE = 'FINISH_FETCHING_PAGE';
export const finishFetchingPage = createAction(FINISH_FETCHING_PAGE);

export const SET_LEAVING_PAGE = 'SET_LEAVING_PAGE';
export const setLeavingPage = createAction(SET_LEAVING_PAGE);

export const SET_ORIGIN_URL = 'SET_ORIGIN_URL';
export const setOriginUrl = createAction(SET_ORIGIN_URL);

export const SET_FULL_URL = 'SET_FULL_URL';
export const setFullUrl = createAction(SET_FULL_URL);

export const SET_LOCATION = 'SET_LOCATION';
export const setLocation = createAction(SET_LOCATION);

function escapeQuery(query) {
    return Object.keys(query).reduce((acc, item) => {
        const escapedKey = encodeSpecialCharacters(item);

        acc[escapedKey] = encodeSpecialCharacters(query[item]);

        return acc;
    }, {});
}

export const setUrlChanges =
    req =>
    ({dispatch}) => {
        let origin;
        let url;
        let location;
        let path;

        if (typeof window !== 'undefined') {
            origin = window.location.origin;
            // Для динамических вещей таких как path и query необходимо использовать req,
            // т.к. мидлвара fullUrl срабатывает раньше чем происходит обновление данных в window.location
            path = encodeSpecialCharacters(req.path);
            url = `${origin}${path}`;
            location = {
                path,
                pathname: req.pathname,
                protocol: window.location.protocol,
                query: escapeQuery(req.query),
            };
        } else {
            let hostname = req.hostname;

            if (process.env.NODE_ENV === 'development') {
                // Для разработки локально сохраняем порт из оригинального хэдера
                const originalHost = req.headers.host;
                // IPv6 literal support
                const offset =
                    originalHost[0] === '[' ? originalHost.indexOf(']') + 1 : 0;
                const portStartIndex = originalHost.indexOf(':', offset);
                const port = parseInt(
                    originalHost.substring(portStartIndex + 1),
                    10,
                );

                if (portStartIndex !== -1 && !Number.isNaN(port)) {
                    hostname = originalHost;
                }
            }

            origin = `${req.protocol}://${hostname}`;
            path = encodeSpecialCharacters(req.url);
            url = `${origin}${path}`;
            location = {
                path,
                protocol: `${req.protocol}:`,
                pathname: req._parsedUrl.pathname,
                query: escapeQuery(req.query),
            };
        }

        batch(() => {
            dispatch(setOriginUrl(origin));
            dispatch(setFullUrl(url));
            dispatch(setLocation(location));
        });
    };
