import {batch} from 'react-redux';

import Platform from '../interfaces/Platform';
import Req from '../interfaces/router/Req';
import Lang from '../interfaces/Lang';
import ISettlement from '../interfaces/state/settlement/ISettlement';
import ThunkAction from '../interfaces/actions/ThunkAction';
import NationalVersion from '../interfaces/NationalVersion';

import {configurePopularDirections} from '../lib/popularDirections';
import {prepareStationsForIndexPage} from '../lib/prepareStationsForIndexPage';
import prepareTeasersForIndexPage from '../lib/prepareTeasersForIndexPage';
import {getWhenForToday} from '../lib/search/contextUtils';

import {addStations} from './stationsGroup';
import {addDirections} from './directions';
import {addPopularDirections} from './popularDirections';
import {setTeasers} from './teasers';
import {setDataForFetchingPage, setBlablacarLink, setIsCity} from './home';
import {
    setFromPointFromUser,
    setToPointFromUser,
    setTransportType,
    setWhen,
} from './searchForm';
import shouldRequestBlablacarCityLink from '../lib/shouldRequestBlablacarCityLink';

const isMobile = process.env.PLATFORM === Platform.mobile;

interface IRequestIndexPage {
    req: Req;
    language: Lang;
    settlement: ISettlement;
    nationalVersion: NationalVersion;
    isCity: boolean;
}

export default function requestHomePage({
    req,
    language,
    settlement,
    nationalVersion,
    isCity,
}: IRequestIndexPage): ThunkAction {
    return async ({dispatch, api, logger}) => {
        const params = {
            language,
            settlementId: settlement.id,
        };

        // сбрасываем поля ввода
        batch(() => {
            dispatch(setDataForFetchingPage());
            dispatch(setFromPointFromUser(settlement));
            dispatch(setToPointFromUser(''));
            dispatch(setWhen(getWhenForToday(language)));
            dispatch(setIsCity(isCity));

            // предзаполняем форму
            const {active_tab} = req.query;

            if (active_tab) {
                dispatch(setTransportType(active_tab));
            }
        });

        const apiRequests = [
            api.execDirections(params, req),
            api.execStations(params, req),
            api.execTeasers(params, req),
            api.execPopularDirections(params, req),
        ];

        if (!isMobile) {
            apiRequests.push(
                shouldRequestBlablacarCityLink(nationalVersion)
                    ? api
                          .execBlablacarCityLink(
                              {
                                  language,
                                  nationalVersion,
                                  id: settlement.id,
                              },
                              req,
                          )
                          .catch(err => {
                              logger.error(
                                  'common/actions/requestIndexPage.ts',
                                  err,
                                  {level: 'error'},
                              );

                              return null;
                          })
                    : Promise.resolve(null),
            );
        }

        const [
            directions,
            {connected, related},
            teasers,
            popularDirections,
            bbkCityLink,
        ] = await Promise.all(apiRequests);

        batch(() => {
            const stations = [...connected, ...related];
            const popularDirectionsConfigured = configurePopularDirections(
                popularDirections,
                req.isBot,
            );

            dispatch(addPopularDirections(popularDirectionsConfigured));

            if (!isMobile) {
                if (bbkCityLink) {
                    dispatch(setBlablacarLink(bbkCityLink));
                }
            }

            dispatch(setTeasers(prepareTeasersForIndexPage(teasers)));
            dispatch(addDirections(directions));
            dispatch(
                addStations(prepareStationsForIndexPage(stations, settlement)),
            );
        });
    };
}
