import {CHAR_EM_DASH, CHAR_NBSP} from '../../lib/stringUtils';

import Req from '../../interfaces/router/Req';
import Direction from '../../interfaces/Direction';
import {FilterTransportType} from '../../lib/transportType';
import ThunkAction from '../../interfaces/actions/ThunkAction';
import StationSubtype from '../../interfaces/state/station/StationSubtype';

import getPointId from '../../lib/point/getPointId';
import {isStation} from '../../lib/point/pointType';
import isAllDaysSearch from '../../lib/search/isAllDaysSearch';
import {getContextFromForm} from '../../lib/search/contextUtils';
import getCanonicalSearchUrl from '../../lib/url/getCanonicalSearchUrl';
import getCanonicalPopularDirectionUrl from '../../lib/url/getCanonicalPopularDirectionUrl';

import {setSearchPopularDirectionsLinks} from './index';

const LINKS_LIMIT = 5;

// Получаем популярные направления для поиска от города
const fetchSettlementPopularDirections =
    (req: Req): ThunkAction =>
    async ({dispatch, api, getState}) => {
        const {searchForm, language, tld} = getState();
        const context = getContextFromForm(searchForm);

        const settlementDirections = await api.execPopularDirections(
            {
                language,
                settlementId: getPointId(context.from.key),
            },
            req,
        );

        if (settlementDirections) {
            const links = settlementDirections.from.points
                .slice(0, LINKS_LIMIT)
                .filter(point => point.slug !== context.to.slug)
                .map(point => ({
                    text: `${context.from.title}${CHAR_NBSP}${CHAR_EM_DASH} ${point.title}`,
                    href: getCanonicalPopularDirectionUrl({
                        point,
                        direction: Direction.from,
                        slug: context.from.slug,
                        language,
                        tld,
                    }),
                }));

            dispatch(setSearchPopularDirectionsLinks(links));
        }
    };

function getStationPopularDirectionsSubtype(
    transportType: FilterTransportType,
): StationSubtype | undefined {
    switch (transportType) {
        case FilterTransportType.train:
            return StationSubtype.train;
        case FilterTransportType.suburban:
            return StationSubtype.suburban;
        case FilterTransportType.plane:
            return StationSubtype.plane;
    }
}

// Получаем популярные направления для поиска от станции
const fetchStationPopularDirections =
    (req: Req): ThunkAction =>
    async ({dispatch, api, getState}) => {
        const {searchForm, language, tld} = getState();
        const context = getContextFromForm(searchForm);

        const stationDirections = await api.execStationPopularDirections2(
            {
                language,
                limit: LINKS_LIMIT,
                stationId: getPointId(context.from.key),
                subtype: getStationPopularDirectionsSubtype(
                    context.transportType,
                ),
            },
            req,
        );

        if (stationDirections) {
            const {from, searchTransportType} = stationDirections;
            const links = from
                .filter(point => point.slug !== context.to.slug)
                .map(point => ({
                    text: `${context.from.title}${CHAR_NBSP}${CHAR_EM_DASH} ${point.title}`,
                    href: getCanonicalSearchUrl({
                        transportType: searchTransportType,
                        toSlug: point.slug,
                        fromSlug: context.from.slug,
                        tld,
                        language,
                    }),
                }));

            dispatch(setSearchPopularDirectionsLinks(links));
        }
    };

const fetchSearchPopularDirections =
    (req: Req): ThunkAction =>
    async ({dispatch, getState, logger}) => {
        try {
            const {searchForm} = getState();
            const context = getContextFromForm(searchForm);

            if (!isAllDaysSearch(context)) {
                return;
            }

            dispatch(setSearchPopularDirectionsLinks([]));

            if (isStation(context.from)) {
                await dispatch(fetchStationPopularDirections(req));
            } else {
                await dispatch(fetchSettlementPopularDirections(req));
            }
        } catch (err) {
            logger.error('common/actions/search/popularDirections.ts', err, {
                req,
                level: 'error',
            });
        }
    };

export default fetchSearchPopularDirections;
