import {batch} from 'react-redux';

import {FilterTransportType} from '../../lib/transportType';

import {TrainPartner} from '../../lib/order/trainPartners';
import EnvironmentType from '../../interfaces/EnvironmentType';

import {getBlablacarData} from './blablacar';
import isAllDaysSearch from '../../lib/search/isAllDaysSearch';
import shouldRequestTrainTariffs from '../../lib/search/shouldRequestTrainTariffs';
import shouldRequestYBus from '../../lib/search/shouldRequestYBus';
import shouldRequestPrices from '../../lib/search/shouldRequestPrices';
import joinStaticPriceAnswers from '../../lib/search/joinStaticPriceAnswers';
import joinDynamicPriceAnswers from '../../lib/search/joinDynamicPriceAnswers';
import getTimeRangeForTrainTariffs from '../../lib/segments/tariffs/getTimeRangeForTrainTariffs';

import {
    setQuerying,
    setBlablacar,
    setStaticTariffs,
    setDynamicTariffs,
    setPricesAreCached,
    setPlaneQueries,
    setTrainPartners,
    setNeedInitialRequestForTrainTariffs,
} from './index';

// Запрашиваем цены
export const requestPrices = ({getState, api, dispatch}, context, req) => {
    const {environment, nationalVersion, clientSettlement, flags, isTouch} =
        getState();
    const {
        transportType,
        language,
        from: {key: pointFrom},
        to: {key: pointTo},
        when: {date: startDate, nextDate: endDate},
    } = context;
    const clientSettlementId = clientSettlement.id;
    const requestsParams = {context, nationalVersion, clientSettlementId};
    const requests = [];

    // по возможности запрашиваем данные блаблакара
    requests.push(
        getBlablacarData(
            api,
            {context, nationalVersion, flags, isMobile: isTouch},
            req,
        ),
    );

    // если контекст поиска позволяет, делаем запрос за электричечными тарифами
    if (
        transportType === FilterTransportType.suburban ||
        transportType === FilterTransportType.all
    ) {
        requests.push(api.exec('suburbanTariffs', requestsParams, req));
    }

    // по возможности запрашиваем цены
    if (shouldRequestPrices(context)) {
        if (isAllDaysSearch(context)) {
            requests.push(api.execAllDaysTariffs(requestsParams, req));
        } else {
            const shouldUseCachedPrices =
                environment.type === EnvironmentType.server;
            const dynamicPriceParams = {
                ...requestsParams,
                poll: shouldUseCachedPrices,
            };

            // цены в поиске на все дни берутся из логов, для них не требуется опрос на клиенте
            dispatch(setPricesAreCached(shouldUseCachedPrices));

            if (
                transportType === FilterTransportType.plane ||
                transportType === FilterTransportType.all
            ) {
                requests.push(
                    api.exec('startPlaneQuerying', dynamicPriceParams, req),
                );
            }
        }

        requests.push(
            api.exec(
                'staticTariffs2',
                {
                    language,
                    nationalVersion,
                    pointFrom,
                    pointTo,
                    startDate,
                    endDate,
                    transportType,
                },
                req,
            ),
        );

        if (shouldRequestYBus(context, nationalVersion)) {
            requests.push(
                api
                    .execBusTariffs2(
                        {
                            context,
                            nationalVersion,
                            environmentType: environment.type,
                            flags,
                        },
                        req,
                    )
                    .then(busTariffsResponse => {
                        dispatch(
                            setQuerying(
                                {bus: busTariffsResponse.busTariffs.querying},
                                getState(),
                            ),
                        );

                        return busTariffsResponse;
                    }),
            );
        }
    }

    return requests;
};

export const requestTrainTariffs =
    req =>
    ({getState, dispatch, api}) => {
        const {
            nationalVersion,
            environment: {type: environmentType},
            flags,
            search,
        } = getState();
        const {context} = search;
        const {
            transportType,
            language,
            from: {key: pointFrom},
            to: {key: pointTo},
            time: {now},
        } = context;

        if (shouldRequestTrainTariffs(context, flags)) {
            const timeRange = getTimeRangeForTrainTariffs(search.segments);

            if (timeRange) {
                return api
                    .exec(
                        'trainTariffs2',
                        {
                            transportType,
                            pointFrom,
                            pointTo,
                            now,
                            language,
                            nationalVersion,
                            flags,
                            poll: false,
                            environmentType,
                            ...timeRange,
                        },
                        req,
                    )
                    .then(({trainTariffs, timeoutError}) => {
                        const {segments, querying} = trainTariffs;

                        if (
                            timeoutError &&
                            environmentType === EnvironmentType.server
                        ) {
                            // Если серверный инициализирующий запрос завершился таймаутом,
                            // то попробуем инициализировать с клиента (RASPFRONT-6374 и RASPFRONT-7187)
                            dispatch(
                                setNeedInitialRequestForTrainTariffs(true),
                            );
                            dispatch(setQuerying({train: true}, getState()));

                            return;
                        }

                        if (segments.length) {
                            dispatch(setDynamicTariffs({segments}, getState()));
                        }

                        dispatch(setQuerying({train: querying}, getState()));
                    });
            }
        }
    };

// раскидываем по stor'у данные о ценах
export const setPrices = (answers, {getState, dispatch}) =>
    batch(() => {
        const {blablacar, planeQIds} = Object.assign({}, ...answers);
        const {currencies, isTouch} = getState();
        const {segments, querying} = joinDynamicPriceAnswers(
            answers,
            currencies,
        );
        const staticTariffs = joinStaticPriceAnswers(answers);

        if (blablacar) {
            dispatch(setBlablacar({blablacar, isMobile: isTouch}));
        }

        if (staticTariffs.length) {
            dispatch(setStaticTariffs(staticTariffs, getState()));
        }

        if (segments.length) {
            dispatch(setDynamicTariffs({segments}, getState()));
        }

        if (planeQIds && Array.isArray(planeQIds.qids)) {
            const planeQueries = planeQIds.qids.map(qid => {
                return {
                    id: qid,
                    skip_partners: [],
                };
            });

            dispatch(setPlaneQueries({planeQueries}, getState()));
        }

        dispatch(setQuerying(querying, getState()));
    });

export const requestTrainPartners =
    ({searchContext, req}) =>
    ({dispatch, getState, logger, api}) => {
        const {transportType} = searchContext;
        const {language} = getState();

        // Зарпашиваем активных партнеров для формирования ссылок
        // на покпку билетов поездов при поиске на все дни.
        // При поиске на конкретную дату ссылки на покупку приходят вместе с тарифами.
        if (
            isAllDaysSearch(searchContext) &&
            [
                FilterTransportType.train,
                FilterTransportType.all,
                FilterTransportType.suburban,
            ].includes(transportType)
        ) {
            return api
                .exec('activeTrainPartners', {language}, req)
                .then(result => dispatch(setTrainPartners(result.partnerCodes)))
                .catch(err => {
                    logger.error(
                        'common/routes/utils/searchGetTrainPartners.js',
                        err,
                    );

                    return dispatch(setTrainPartners([TrainPartner.ufs]));
                });
        }
    };
