import {batch} from 'react-redux';

import {MOBILE} from '../../lib/platforms';

import {FilterTransportType} from '../../lib/transportType';

import getTransportTypeForRequestTransfer from '../../lib/search/getTransportTypeForRequestTransfer';
import {
    getSearchContext,
    getExtendedSearchContext,
    getContextFromForm,
} from '../../lib/search/contextUtils';
import shouldRequestSuburbanTransfer from '../../lib/search/shouldRequestSuburbanTransfer';

import {requestPrices, requestTrainTariffs, setPrices} from './prices';
import {addItemToSearchHistory} from '../searchHistory';
import {
    setPlans,
    setTeasers,
    setSegments,
    setSearchContext,
    upsertTransferSegments,
    setEmptySearchResult,
    setTransferQuerying,
    setArchivalData,
    setCanonicals,
    setBannerInfo,
} from './index';
import requestTransfer from '../../lib/search/requestTransfer';
import joinTransferAnswers from '../../lib/search/joinTransferAnswers';
import searchAddParamsOfVisit from '../../routes/utils/searchAddParamsOfVisit';
import isAllDaysSearch from '../../lib/search/isAllDaysSearch';
import {flagsSelector} from '../../selectors/flagsSelector';

export const performTransfersRequest =
    (context, req, checker) =>
    async ({dispatch, api, getState}) => {
        const state = getState();
        const {
            tld,
            flags,
            user: {isBot},
            platform,
        } = state;
        const {
            transportType,
            from: {key: pointFrom},
            to: {key: pointTo},
            when: {date: when},
            language,
        } = context;

        const transferAnswer = await requestTransfer({
            transportType,
            pointFrom,
            pointTo,
            when,
            language,
            tld,
            poll: false,
            req,
            flags,
            platform,
            api,
            isBot,
        });

        await checker();

        const {transfers, status} = joinTransferAnswers([transferAnswer]);

        if (transfers && transfers.length) {
            batch(() => {
                dispatch(upsertTransferSegments(transfers, getState()));
                dispatch(setTransferQuerying(status));
            });
        }
    };

const buildRequest = ({dispatch, getState, api}, req) =>
    batch(() => {
        const {
            flags,
            tld,
            platform,
            user: {isBot},
            searchForm,
            nationalVersion,
        } = getState();
        const isTrainPartnerTesting = Boolean(flags.__ufsTesting);
        const context = getContextFromForm(searchForm);
        const {
            from: {key: pointFrom},
            to: {key: pointTo},
            when: {date: when},
            language,
            transportType,
        } = context;

        dispatch(addItemToSearchHistory(context));

        const suburbanTransfers = shouldRequestSuburbanTransfer(context)
            ? requestTransfer({
                  transportType: FilterTransportType.suburban,
                  pointFrom,
                  pointTo,
                  when,
                  language,
                  tld,
                  poll: false,
                  req,
                  flags,
                  platform,
                  api,
                  isBot,
              })
            : {};

        return [
            api.execSearch(
                {
                    context,
                    isMobile: platform === MOBILE,
                    excludeTrains: isTrainPartnerTesting,
                    nationalVersion,
                    groupTrains:
                        isAllDaysSearch(context) &&
                        transportType === FilterTransportType.train,
                    allowChangeContext: !isTrainPartnerTesting,
                },
                req,
            ),
            suburbanTransfers,
            ...requestPrices({getState, api, dispatch}, context, req),
            api.execCanonicals(
                {pointFrom, pointTo, transportType, language},
                req,
            ),
        ];
    });

const dispatchRequest = (
    answers,
    context,
    {getState, dispatch},
    req,
    checker,
) =>
    batch(() => {
        const {search} = Object.assign({}, ...answers);
        const {transfers, transportTypes, status} =
            joinTransferAnswers(answers);

        dispatch(setSearchContext(context));
        // добавляем параметры визита в метрику
        searchAddParamsOfVisit(getState());
        dispatch(setSegments([...search.segments, ...transfers], getState()));
        dispatch(setTeasers(search.teasers || {}));
        dispatch(setPlans(search.plans || {}));
        dispatch(setBannerInfo(search.bannerInfo));

        if (Object.keys(status)) {
            dispatch(setTransferQuerying(status));
        }

        setPrices(answers, {dispatch, getState});

        const flags = flagsSelector(getState());
        const transferTransportType = getTransportTypeForRequestTransfer(
            getState().search,
            flags,
        );

        if (
            transferTransportType &&
            !transportTypes.includes(transferTransportType)
        ) {
            const transferContext = {
                ...context,
                transportType: transferTransportType,
            };

            return dispatch(
                performTransfersRequest(transferContext, req, checker),
            );
        }
    });

const dispatchRequestWithChangedContext = (
    search,
    context,
    store,
    req,
    checker,
) => {
    return Promise.all(requestPrices(store, context, req))
        .then(checker)
        .then(prices =>
            dispatchRequest(
                [...prices, {search}],
                context,
                store,
                req,
                checker,
            ),
        );
};

const dispatchTransferRequest = (answers, context, {getState, dispatch}) => {
    const {search} = Object.assign({}, ...answers);
    const {transfers} = joinTransferAnswers(answers);

    dispatch(setSearchContext(context));
    dispatch(setSegments(transfers, getState()));
    dispatch(setTeasers(search.teasers || {}));
    dispatch(setPlans(search.plans || {}));
    dispatch(setBannerInfo(search.bannerInfo));
};

const parseRequest = (answers, store, req, checker) => {
    const {dispatch, getState} = store;
    const state = getState();

    const {search, canonicals} = Object.assign({}, ...answers);
    const {transfers} = joinTransferAnswers(answers);

    dispatch(setArchivalData(search.archivalData));
    dispatch(setCanonicals(canonicals));

    const context = getSearchContext(state.searchForm, search);

    if (!search.context.isChanged) {
        return dispatchRequest(answers, context, store, req, checker);
    }

    // произошло сужение/расширение в контексе поиска в ручке search

    if (transfers && transfers.length) {
        // Если были найдены пересадки, то показываем только их
        return dispatchTransferRequest(answers, context, store);
    }

    const changedContext = getExtendedSearchContext(context, search.context);

    // перезапрашиваем цены для изменившегося контекса
    return dispatchRequestWithChangedContext(
        search,
        changedContext,
        store,
        req,
        checker,
    );
};

export const performSearchRequest = (req, checker) => store => {
    return Promise.all(buildRequest(store, req))
        .then(checker)
        .then(answers => parseRequest(answers, store, req, checker))
        .then(checker)
        .then(() => store.dispatch(requestTrainTariffs(req)));
};

export const noResults =
    () =>
    ({dispatch, getState}) =>
        dispatch(
            setEmptySearchResult(getContextFromForm(getState().searchForm)),
        );
