import {createAction} from 'redux-actions';
import page from 'page';
import upperFirst from 'lodash/upperFirst';
import omit from 'lodash/omit';

import {TRANSPORT_TYPE_BY_TYPE_IN_REQUEST} from '../lib/transportType';
import {LASTOCHKA} from '../routes/searchWithSlugs';
import {SPECIAL_VALUES} from '../lib/date/values';
import {ParseError, ERROR_TYPES} from '../lib/errors/errorTypes';

import DateSpecialValue from '../interfaces/date/DateSpecialValue';
// eslint-disable-next-line no-duplicate-imports
import {FilterTransportType} from '../lib/transportType';

import compareSearchContexts from '../lib/compareSearchContexts';
import parseDate from '../lib/date/parse';
import {getParseParams} from '../lib/date/utils';
import isAllDaysSearch from '../lib/search/isAllDaysSearch';
import searchUrl, {needUseUrlWithSlug} from '../lib/url/searchUrl';
import {validateWhen} from '../lib/search/contextUtils';

export const SET_FROM = Symbol('SET_FROM');
export const setFrom = createAction(SET_FROM);

export const SET_ORIGINAL_FROM = Symbol('SET_ORIGINAL_FROM');
export const setOriginalFrom = createAction(SET_ORIGINAL_FROM);

export const SET_TO = Symbol('SET_TO');
export const setTo = createAction(SET_TO);

export const SET_ORIGINAL_TO = Symbol('SET_ORIGINAL_TO');
export const setOriginalTo = createAction(SET_ORIGINAL_TO);

export const SET_WHEN = Symbol('SET_WHEN');
export const setWhen = createAction(SET_WHEN);

export const SWAP = Symbol('SWAP');
export const swap = createAction(SWAP);

export const SET_TRANSPORT_TYPE = Symbol('SET_TRANSPORT_TYPE');
export const setTransportType = createAction(SET_TRANSPORT_TYPE);

export const SET_TRANSPORT_TYPE_FROM_USER = Symbol(
    'SET_TRANSPORT_TYPE_FROM_USER',
);
export const setTransportTypeFromUser = createAction(
    SET_TRANSPORT_TYPE_FROM_USER,
);

export const SET_ERRORS = Symbol('SET_ERRORS');
export const setErrors = createAction(SET_ERRORS);

export const SET_FORM_DATA = Symbol('SET_FORM_DATA');
export const setFormData = createAction(SET_FORM_DATA);

export const SET_USER_INPUT = Symbol('SET_USER_INPUT');
export const setUserInput = createAction(SET_USER_INPUT);

export const SET_FROM_POINT_FROM_USER = Symbol('SET_FROM_POINT_FROM_USER');
export const setFromPointFromUser = createAction(SET_FROM_POINT_FROM_USER);

export const SET_TO_POINT_FROM_USER = Symbol('SET_TO_POINT_FROM_USER');
export const setToPointFromUser = createAction(SET_TO_POINT_FROM_USER);

export const START_VALIDATION_PROCESSING = 'START_VALIDATION_PROCESSING';
export const startValidationProcessing = createAction(
    START_VALIDATION_PROCESSING,
);

export const FINISH_VALIDATION_PROCESSING = 'FINISH_VALIDATION_PROCESSING';
export const finishValidationProcessing = createAction(
    FINISH_VALIDATION_PROCESSING,
);

function updateUserInput(getState, dispatch, parseContextResponse) {
    const {searchForm} = getState();
    const {from: userInputFrom = {}, to: userInputTo = {}} =
        searchForm.userInput || {};
    const {from, originalFrom, to, originalTo} = parseContextResponse;
    const newUserInput = {};

    if (!userInputFrom.title) {
        newUserInput.from = originalFrom || from;
    } else if (
        !userInputFrom.slug &&
        originalFrom.slug &&
        userInputFrom.key &&
        userInputFrom.key === originalFrom.key
    ) {
        newUserInput.from = {
            ...userInputFrom,
            slug: originalFrom.slug,
        };
    }

    if (!userInputTo.title) {
        newUserInput.to = originalTo || to;
    } else if (
        !userInputTo.slug &&
        originalTo.slug &&
        userInputTo.key &&
        userInputTo.key === originalTo.key
    ) {
        newUserInput.to = {
            ...userInputTo,
            slug: originalTo.slug,
        };
    }

    if (Object.keys(newUserInput).length) {
        dispatch(setUserInput(newUserInput));
    }
}

export const setFormDataFromRequest =
    (query, params, searchNext) =>
    ({dispatch, getState}) => {
        const searchForm = getState().searchForm;
        const {
            originalFrom,
            originalTo,
            from,
            to,
            transportType: currentTransportType,
        } = searchForm;
        const fromInRequest = {
            title: query.fromName,
            key: query.fromId,
        };
        const toInRequest = {
            title: query.toName,
            key: query.toId,
        };

        const when = parseDate(
            query.when || '',
            getParseParams(searchForm),
            true,
        );
        const plan =
            isAllDaysSearch({when}) &&
            params.transportType === FilterTransportType.suburban &&
            query.plan
                ? query.plan
                : null;
        const transportType =
            TRANSPORT_TYPE_BY_TYPE_IN_REQUEST[
                params.transportType || query.transportType
            ] || FilterTransportType.all;

        const fromAreEqual =
            currentTransportType === transportType &&
            originalFrom.title === fromInRequest.title &&
            originalFrom.key === fromInRequest.key;
        const toAreEqual =
            currentTransportType === transportType &&
            originalTo.title === toInRequest.title &&
            originalTo.key === toInRequest.key;

        const formData = {
            originalFrom: fromAreEqual ? originalFrom : fromInRequest,
            originalTo: toAreEqual ? originalTo : toInRequest,
            from: fromAreEqual ? from : fromInRequest.title,
            to: toAreEqual ? to : toInRequest.title,
            transportType,
            when,
            searchNext,
            plan,
        };

        dispatch(setFormData(formData));
        dispatch(
            setUserInput({
                from: fromInRequest,
                to: toInRequest,
            }),
        );
    };

export const setFormDataFromRequestWithSlugs =
    (query, params) =>
    ({dispatch, getState}) => {
        const {searchForm, language} = getState();
        const {fromSlug, toSlug, name} = params;
        const fromInRequest = {slug: fromSlug};
        const toInRequest = {slug: toSlug};
        const {
            from,
            to,
            originalFrom,
            originalTo,
            userInput,
            transportType: currentTransportType,
        } = searchForm;
        const isLastochka = name === LASTOCHKA;

        const transportType = params.transportType
            ? TRANSPORT_TYPE_BY_TYPE_IN_REQUEST[params.transportType]
            : isLastochka
            ? FilterTransportType.suburban
            : FilterTransportType.all;
        const date =
            params.when === DateSpecialValue.today || isLastochka
                ? SPECIAL_VALUES[language][DateSpecialValue.today]
                : SPECIAL_VALUES[language][DateSpecialValue.allDays];
        const when = parseDate(date, getParseParams(searchForm), true);
        const plan =
            transportType === FilterTransportType.suburban && query.plan
                ? query.plan
                : null;

        const transportTypesAreEqual = currentTransportType === transportType;
        const fromAreEqual =
            originalFrom.slug === fromInRequest.slug && transportTypesAreEqual;
        const toAreEqual =
            originalTo.slug === toInRequest.slug && transportTypesAreEqual;

        const formData = {
            originalFrom: fromAreEqual ? originalFrom : fromInRequest,
            originalTo: toAreEqual ? originalTo : toInRequest,
            from: fromAreEqual ? from : {},
            to: toAreEqual ? to : {},
            transportType,
            when,
            plan,
        };

        dispatch(setFormData(formData));

        const newUserInput = {};

        if (fromInRequest.slug === userInput.from.slug) {
            newUserInput.from = userInput.from;
        } else if (fromAreEqual) {
            newUserInput.from = originalFrom;
        } else {
            newUserInput.from = fromInRequest;
        }

        if (toInRequest.slug === userInput.to.slug) {
            newUserInput.to = userInput.to;
        } else if (toAreEqual) {
            newUserInput.to = originalTo;
        } else {
            newUserInput.to = toInRequest;
        }

        dispatch(setUserInput(newUserInput));
    };

export const validate =
    (req = {}) =>
    ({getState, dispatch, api}) => {
        const {tld, language, searchForm} = getState();
        const {
            transportType,
            originalFrom: {slug: fromSlug, key: fromKey, title: fromTitle},
            originalTo: {slug: toSlug, key: toKey, title: toTitle},
        } = searchForm;

        return api
            .exec(
                'parseContext',
                {
                    tld,
                    language,
                    transportType,
                    fromSlug,
                    fromKey,
                    fromTitle,
                    toSlug,
                    toKey,
                    toTitle,
                },
                req,
            )
            .then(responseApi => {
                const currentSearchForm = getState().searchForm;
                const {searchNext, plan, when} = currentSearchForm;
                const response = {
                    ...responseApi,
                    when,
                    errors: [...responseApi.errors, ...validateWhen(when)],
                };

                if (compareSearchContexts(searchForm, currentSearchForm)) {
                    if (!response.errors.length) {
                        dispatch(setFormData({...response, searchNext, plan}));
                        updateUserInput(getState, dispatch, response);
                    } else {
                        // если в ответе есть ошибки для полей Откуда или Куда, то берем эти поля из стора, а не из ответа
                        const {errors} = response;

                        const newFromTo = {
                            to: response.to,
                            from: response.from,
                            originalTo: response.originalTo,
                            originalFrom: response.originalFrom,
                        };

                        errors.forEach(({fields = []}) => {
                            fields.forEach(field => {
                                if (field !== 'to' && field !== 'from') {
                                    return;
                                }

                                newFromTo[field] = currentSearchForm[field];

                                const originalField = `original${upperFirst(
                                    field,
                                )}`;

                                newFromTo[originalField] =
                                    currentSearchForm[originalField];
                            });
                        });

                        dispatch(
                            setFormData({
                                ...response,
                                ...newFromTo,
                                searchNext,
                                plan,
                            }),
                        );

                        throw new ParseError(
                            'Invalid search request',
                            response.errors.map(error => error.type),
                        );
                    }
                } else {
                    throw new Error('Obsolete search form validation request');
                }
            })
            .catch(err => {
                if (
                    err instanceof ParseError === false &&
                    err.statusCode === 500
                ) {
                    // в случае, если ручка апи отдает 500, отдаем пользователю 404, и
                    // показываем пустую страницу поиска
                    throw new ParseError(err.message, [ERROR_TYPES.NO_ROUTES]);
                }

                throw err;
            });
    };

export const getSuburbanWidgetSearchContext =
    req =>
    ({getState, dispatch, api}) => {
        const {nationalVersion, clientSettlement, searchForm} = getState();

        return api
            .exec(
                'suburbanWidgetSearchContext',
                {
                    nationalVersion,
                    searchForm,
                    clientSettlementId: clientSettlement.id,
                },
                req,
            )
            .then(response => {
                const currentSearchForm = getState().searchForm;
                const {searchNext, plan} = currentSearchForm;

                if (compareSearchContexts(searchForm, currentSearchForm)) {
                    if (!response.errors.length) {
                        dispatch(setFormData({...response, searchNext, plan}));
                        updateUserInput(getState, dispatch, response);
                    } else {
                        dispatch(setFormData({...response, searchNext, plan}));
                        throw new ParseError(
                            'Invalid search request',
                            response.errors.map(error => error.type),
                        );
                    }
                } else {
                    throw new Error('Obsolete search form validation request');
                }
            });
    };

/**
 * Добавляет слаги к полям пользовательского ввода.
 * Нужно для того, чтобы понимать, что userInput соответствует контексту поиска
 * @return {Function}
 */
export const addSlugsToUserInput =
    () =>
    ({dispatch, getState}) => {
        const {searchForm} = getState();
        const {userInput, originalFrom, originalTo} = searchForm;

        dispatch(
            setUserInput({
                from: {
                    ...userInput.from,
                    slug: originalFrom.slug,
                },
                to: {
                    ...userInput.to,
                    slug: originalTo.slug,
                },
            }),
        );
    };

export const submitForm =
    () =>
    ({dispatch, getState}) => {
        const {searchForm, tld, language} = getState();

        dispatch(startValidationProcessing('process'));

        return Promise.resolve()
            .then(() => dispatch(validate()))
            .then(() => {
                const {searchForm: newSearchForm} = getState();

                if (needUseUrlWithSlug(newSearchForm)) {
                    // добавляем слаги к полям userInput для того,
                    // чтобы потом понять, что данные в userInput соответствуют контексту
                    // и их не нужно обновлять
                    dispatch(addSlugsToUserInput());
                }

                return page.show(
                    searchUrl(
                        {context: omit(newSearchForm, 'plan')},
                        tld,
                        language,
                    ),
                );
            })
            .catch(() => {
                // показываем ошибку по старому урлу
                const path = searchUrl(
                    {
                        context: {
                            ...omit(searchForm, 'plan'),
                            when: {
                                ...searchForm.when,
                                special: '',
                            },
                        },
                    },
                    tld,
                    language,
                );

                return page.show(path);
            })
            .finally(() => {
                dispatch(finishValidationProcessing());
            });
    };
