import React, {ReactElement, PureComponent, ChangeEvent} from 'react';
import B from 'bem-cn-lite';
import {connect, DispatchProp} from 'react-redux';

import BugType from '../../interfaces/state/bugReporter/BugType';
import IApiBugReportRequest from '../../interfaces/api/IApiBugReportRequest';
import IStateBugReporter from '../../interfaces/state/bugReporter/IStateBugReporter';
import IState from '../../interfaces/state/IState';
import IconGlyph from '../../interfaces/components/IconGlyph';

import {
    sendBugReport,
    setTitle,
    setDescription,
    setFormIsOpened,
    setBugType,
} from '../../actions/bugReporter';

import Button2 from '../Button2/Button2';
import Input from '../Input/Input';
import Modal from '../basic/Modal';
import RadioGroup from '../basic/RadioGroup';
import Icon from '../Icon/Icon';

import keyset from '../../i18n/bug-reporter';

const b = B('BugReporter');

const meta = [
    {
        title: 'url',
        collector() {
            return location.href;
        },
    },
    {
        title: 'windowSize',
        collector() {
            return {
                width: window.outerWidth,
                height: window.outerHeight,
            };
        },
    },
    {
        title: 'screenSize',
        collector() {
            return {
                width: screen.width,
                height: screen.height,
            };
        },
    },
    {
        title: 'userAgent',
        collector() {
            return navigator.userAgent;
        },
    },
    {
        title: 'cookies',
        collector() {
            return document.cookie;
        },
    },
];

const problemTypes: Record<BugType, string> = {
    [BugType.interfaceBug]: keyset('problem-types-interface-bug'),
    [BugType.dataBug]: keyset('problem-types-data-bug'),
    [BugType.performanceBug]: keyset('problem-types-performance-bug'),
    [BugType.typeBug]: keyset('problem-types-type-bug'),
    [BugType.otherBug]: keyset('problem-types-other-bug'),
};

function mapStateToProps(state: IState): IStateBugReporter {
    return state.bugReporter;
}

class BugReporter extends PureComponent<IStateBugReporter & DispatchProp> {
    onTitleChange = (newValue: string): void => {
        this.props.dispatch(setTitle(newValue));
    };

    onDescriptionChange = (newValue: string): void => {
        this.props.dispatch(setDescription(newValue));
    };

    onRadioChange = (event: ChangeEvent<HTMLInputElement>): void => {
        this.props.dispatch(setBugType(event.target.value as BugType)); // Простите за "as", но прямо сейчас компонент RadioGroup не типизирован и использует data-атрибуты
    };

    onModalClose = (): void => {
        this.props.dispatch(setFormIsOpened(false));
    };

    onToggleClick = (): void => this.toggleOpen();

    onSubmitClick = (): void =>
        this.props.dispatch(sendBugReport(this.collectData()));

    collectData(): IApiBugReportRequest {
        const {title, description, bugType} = this.props;

        return {
            title,
            description,
            problem: problemTypes[bugType],
            meta: meta.reduce((acc, item) => {
                acc[item.title] = item.collector();

                return acc;
            }, {}),
        };
    }

    toggleOpen(): void {
        this.props.dispatch(setFormIsOpened(!this.props.formIsOpened));
    }

    renderBug(): ReactElement {
        return <Button2 className={b('button')} onClick={this.onToggleClick} />;
    }

    renderRadio(): ReactElement {
        const radios = Object.keys(problemTypes).map(key => {
            const text = problemTypes[key];

            return {value: key, name: key, label: text};
        }, []);

        return (
            <RadioGroup
                className={b('radiogroup')}
                view="default"
                theme="normal"
                size="m"
                name="problemTypes"
                value={this.props.bugType}
                onChange={this.onRadioChange}
                buttons={radios}
            />
        );
    }

    renderContentForm(): ReactElement {
        const {reportSending, title, description, bugType} = this.props;

        const disableCancelButton = reportSending;
        const disableConfirmButton =
            !title || !description || !bugType || reportSending;

        return (
            <>
                {this.renderRadio()}

                <div>
                    <div className={b('required')}>
                        {keyset('form-title-placeholder')}
                    </div>

                    <Input
                        value={title}
                        onChange={this.onTitleChange}
                        widthMod="100"
                    />
                </div>

                <div>
                    <div className={b('required')}>
                        {keyset('form-textarea-placeholder')}
                    </div>

                    <Input
                        value={description}
                        onChange={this.onDescriptionChange}
                        rows={5}
                        textarea
                        widthMod="100"
                    />
                </div>

                <div className={b('actions')}>
                    <Button2
                        className={b('actionButton')}
                        theme="action"
                        disabled={disableConfirmButton}
                        type="submit"
                        onClick={this.onSubmitClick}
                    >
                        {keyset('submit-button-text')}
                    </Button2>

                    <Button2
                        className={b('actionButton')}
                        theme="action"
                        disabled={disableCancelButton}
                        type="reset"
                        onClick={this.onToggleClick}
                    >
                        {keyset('cancel-button-text')}
                    </Button2>

                    {this.renderStatus()}
                </div>
            </>
        );
    }

    renderForm(): ReactElement | null {
        const {formIsOpened, reportIsSent} = this.props;

        if (!formIsOpened) {
            return null;
        }

        return (
            <Modal
                onClose={this.onModalClose}
                hasCloseButton={false}
                mods={{paddings: 'none', content: 'column'}}
                visible={formIsOpened}
            >
                <div className={b('title')}>
                    {keyset('form-title')}

                    <span
                        className={b('closeContainer')}
                        onClick={this.onToggleClick}
                    >
                        <Icon
                            className={b('closeIcon')}
                            glyph={IconGlyph.cross}
                        />
                    </span>
                </div>

                <div className={b('content')}>
                    {reportIsSent
                        ? this.renderStatus()
                        : this.renderContentForm()}
                </div>
            </Modal>
        );
    }

    renderStatus(): ReactElement | null {
        if (this.props.reportIsSent === null) {
            return null;
        }

        return (
            <div className={b('status')}>
                {keyset(
                    this.props.reportIsSent
                        ? 'submit-ok-message'
                        : 'submit-fail-message',
                )}
            </div>
        );
    }

    render(): ReactElement {
        return (
            <div className={b()}>
                {this.renderBug()}
                {this.renderForm()}
            </div>
        );
    }
}

export default connect(mapStateToProps)(BugReporter);
