import React, {
    memo,
    ReactElement,
    SyntheticEvent,
    useState,
    useCallback,
    PropsWithChildren,
} from 'react';
import B from 'bem-cn-lite';

import Platform from '../../interfaces/Platform';

import Link from '../Link';

const b = B('Button2');

const DEFAULT_SIZE_MOD = process.env.PLATFORM === Platform.mobile ? 'l' : 'm';

interface IButtonCommon {
    className?: string;
    iconLeft?: string | ReactElement;
    iconRight?: string | ReactElement;
    borderRadius?: 'none' | 'left' | 'right';
    // select - кнопка для селекта с обводкой по контуру
    // normal - по умолчанию
    theme?: 'normal' | 'action' | 'select' | 'disabled';
    ellipsis?: boolean; // Текст внутри может обрезаться с "..." на конце
    minWidth?: number; // Минимальная ширина кнопки. Имеет смысл использовать совместно с "ellipsis: true"
    selected?: boolean;
    disabled?: boolean;
    rightSideButtonDelimiter?: boolean;
    onClick?: (e: SyntheticEvent<HTMLElement>) => void;
    sizeMod?: 'm' | 'l' | 'filterButton';
    textAlign?: 'left' | 'center' | 'right';
}

type ButtonCommon = PropsWithChildren<IButtonCommon>;

interface IButtonLink extends ButtonCommon {
    url: string;
}

interface IButton extends ButtonCommon {
    type?: 'button' | 'reset' | 'submit';
    filterButton?: boolean;
    disabled?: boolean;
}

export default memo(Button2);

function Button2(props: IButtonLink): ReactElement;
function Button2(props: IButton): ReactElement;
function Button2(props: IButtonLink | IButton): ReactElement {
    const {
        children,
        className = '',
        disabled,
        theme = 'normal',
        selected,
        borderRadius,
        iconLeft,
        iconRight,
        rightSideButtonDelimiter = false,
        onClick,
        ellipsis,
        minWidth,
        sizeMod = DEFAULT_SIZE_MOD,
        textAlign,
    } = props;

    const [pressed, setPressed] = useState(false);

    const onPressed = useCallback(() => setPressed(true), []);
    const onReleased = useCallback(() => setPressed(false), []);

    // Грязный хак для data-атрибутов. Если в данный компонент они больше не передаются, то срочно выпилите это, пожалуйста, иначе моя душа и после смерти не сможет найти себе покоя и я буду являться вам по ночам в образе неуловимого бага, который порождает утечки памяти в вашем продакшене
    const dataAttributes = Object.entries(props)
        .filter(([name]) => name.startsWith('data-'))
        .reduce((result, [key, value]) => {
            result[key] = value;

            return result;
        }, {});

    const commonAttributes = {
        onMouseDown: onPressed,
        onTouchStart: onPressed,

        onMouseUp: onReleased,
        onTouchEnd: onReleased,

        onClick,
        style: {
            minWidth,
        },

        ...dataAttributes,
    };

    const commonMods = {
        selected,
        theme: disabled ? 'disabled' : theme,
        pressed,
        borderRadius,
        rightSideButtonDelimiter,
        iconLeft: Boolean(iconLeft),
        iconRight: Boolean(iconRight),
        ellipsis,
        size: sizeMod,
        textAlign,
    };

    const content = (
        <>
            {iconLeft && <i className={b('icon', {left: true})}>{iconLeft}</i>}

            <span className={b('content')}>{children}</span>

            {iconRight && (
                <i className={b('icon', {right: true})}>{iconRight}</i>
            )}
        </>
    );

    if ('url' in props) {
        // Кнопка-ссылка
        const {url} = props;

        return (
            <Link
                {...commonAttributes}
                href={url}
                className={b(commonMods, className)}
            >
                {content}
            </Link>
        );
    }

    // Кнопка
    const {type = 'button', filterButton} = props;

    return (
        <button
            {...commonAttributes}
            type={type}
            disabled={disabled}
            className={b(
                {
                    ...commonMods,
                    filterButton,
                },
                className,
            )}
        >
            {content}
        </button>
    );
}
