import {React, PureComponent, B} from '../base';

import {SyntheticEvent} from 'react';

import ITime from '../../interfaces/state/searchForm/ITime';
import Lang from '../../interfaces/Lang';
import IFlags from '../../interfaces/state/flags/IStateFlags';
import Tld from '../../interfaces/Tld';
import IWhen from '../../interfaces/state/IWhen';
import IMergedMaskDaysOfWalking from '../../interfaces/maskDaysOfWalking/IMergedMaskDaysOfWalking';

import {reachGoal} from '../../lib/yaMetrika';
import {buildFromObject} from '../../lib/date/build';
import {
    getToday,
    getRange,
    extendRangeToArray,
    getParseParams,
} from '../../lib/date/utils';

import CalendarMonth from '../CalendarMonth/CalendarMonth';
import CalendarWeekday from '../CalendarWeekday/CalendarWeekday';
import CalendarScroller from '../CalendarScroller/CalendarScroller';

const b = B('Calendar');

interface IError {
    title?: string;
    type?: string;
}

interface ICalendarProps {
    id: string;
    time: ITime;
    value: IWhen;
    language: Lang;
    onClick: (e: SyntheticEvent<HTMLElement>, data?: object) => void;
    onMouseOut: (e: SyntheticEvent) => void;
    onMouseOver: (e: SyntheticEvent, date?: object) => void;
    flags: IFlags;
    tld: Tld;

    mask?: IMergedMaskDaysOfWalking;
    errors?: IError[];
}

export default class Calendar extends PureComponent<ICalendarProps, {}> {
    onMouseDown(e: SyntheticEvent): void {
        e.preventDefault();
    }

    onClick = (e: SyntheticEvent<HTMLElement>): void => {
        const {dataset} = e.currentTarget;
        const {date, special, weekday} = dataset;
        const value = buildFromObject(
            {
                date,
                special,
                weekday,
            },
            getParseParams(this.props),
        );

        if (value) {
            if (value.special) {
                reachGoal('calendar_toolbar_button_click', {
                    button: value.special,
                });
            }

            this.props.onClick(e, {value});
        }
    };

    onMouseOver = (e): void => {
        const {time, language, tld} = this.props;
        const value = buildFromObject(
            e.target.dataset,
            getParseParams({
                time,
                language,
                tld,
            }),
        );
        const hint = value && value.formatted;

        if (hint) {
            this.props.onMouseOver(e, {hint});
        }
    };

    render(): React.ReactElement {
        const {
            id,
            time,
            value,
            mask,
            onMouseOut,
            language,
            errors = [],
        } = this.props;

        const today = getToday(time);
        const range = getRange(today);
        const months = extendRangeToArray(range, 'month', 'month');

        return (
            <label className={b()} onMouseDown={this.onMouseDown} htmlFor={id}>
                {errors.map(error => (
                    <div className={b('error')} key={error.type}>
                        {error.title}
                    </div>
                ))}

                <CalendarScroller
                    time={time}
                    mask={mask}
                    value={value}
                    today={today}
                    range={range}
                    months={months}
                    language={language}
                    isRedesigned
                    onClick={this.onClick}
                    onMouseOut={onMouseOut}
                    onMouseOver={this.onMouseOver}
                    MonthComponent={CalendarMonth}
                    WeekdayComponent={CalendarWeekday}
                />
            </label>
        );
    }
}
