import {refCallback, PropTypes} from '../base';
import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';

import {ROBOT} from '../../lib/date/formats';

import {reachGoal} from '../../lib/yaMetrika';
import {buildFromObject} from '../../lib/date/build';
import {
    getToday,
    getRange,
    extendRangeToArray,
    getParseParams,
} from '../../lib/date/utils';
import noop from '../../lib/noop';

import Modal from '../basic/Modal';
import Month from '../Month/Month';
import CalendarDay from '../CalendarDay';
import CalendarToolbar from '../CalendarToolbar/CalendarToolbar';
import ModalHistoryWrapper from '../ModalHistoryWrapper/ModalHistoryWrapper';

import keyset from '../../i18n/search-calendar';

const b = B('Calendar');

export default class Calendar extends PureComponent {
    static propTypes = {
        time: PropTypes.time.isRequired,
        value: PropTypes.datePickerValue.isRequired,
        language: PropTypes.language.isRequired,

        onChange: PropTypes.func,
        opened: PropTypes.bool,
        onClose: PropTypes.func,
        onClick: PropTypes.func,
        fixed: PropTypes.bool,
        showToolbar: PropTypes.bool,
        DayComponent: PropTypes.elementType,
        months: PropTypes.array,
        flags: PropTypes.object,
    };

    static defaultProps = {
        showToolbar: true,
        fixed: true,
        DayComponent: CalendarDay,
        opened: false,
        onChange: noop,
        onClose: noop,
        onClick: noop,
        flags: {},
    };

    onCloseCaendar = () => this.props.onClose;

    onShow = () => {
        // Скроллим календарь на текущий месяц
        const currentMonthIndex = this.getCurrentMonthIndex();
        const {top: containerTop} =
            this._monthsContainer.getBoundingClientRect();
        let currentMonthTop = null;

        if (this._monthsContainer?.childNodes[currentMonthIndex]) {
            currentMonthTop =
                this._monthsContainer.childNodes[
                    currentMonthIndex
                ].getBoundingClientRect().top;
        }

        if (this._monthsContainer && currentMonthTop !== null) {
            this._monthsContainer.scrollTop = currentMonthTop - containerTop;
        }
    };

    onClick = e => {
        const {onClose, onClick: propsOnClick, onChange} = this.props;
        const value = buildFromObject(
            e.target.dataset,
            getParseParams(this.props),
        );

        if (value) {
            if (value.special) {
                reachGoal('calendar_toolbar_button_click', {
                    button: value.special,
                });
            }

            propsOnClick();
            onChange(e, {value});
            onClose();
        }
    };

    getCurrentMonthIndex() {
        const {time, months: propsMonths} = this.props;

        const today = getToday(time);
        const formattedToday = today.format('YYYY-MM');
        const months =
            propsMonths ||
            extendRangeToArray(getRange(today), 'month', 'month');

        return months.findIndex(
            month => month.format('YYYY-MM') === formattedToday,
        );
    }

    getCalendar() {
        const {
            time,
            value,
            opened,
            showToolbar,
            DayComponent,

            context,
            language,

            fixed,
            onClose,
            ...props
        } = this.props;

        const {flags} = props;

        const today = getToday(time);
        const range = getRange(today);
        const {months = extendRangeToArray(range, 'month', 'month')} =
            this.props;

        return (
            <Modal
                className={b()}
                title={keyset('title')}
                fixed={fixed}
                visible={opened}
                onClose={onClose}
                onShow={this.onShow}
            >
                {showToolbar && (
                    <CalendarToolbar
                        className={b('toolbar')}
                        value={value}
                        time={time}
                        language={language}
                        onClick={this.onClick}
                        flags={flags}
                    />
                )}

                <div
                    className={b('months')}
                    ref={refCallback(this, '_monthsContainer')}
                >
                    {months.map((month, i) => (
                        <div className={b('month')} key={month.format(ROBOT)}>
                            <Month
                                {...props}
                                context={context}
                                language={language}
                                value={value}
                                month={month}
                                today={today}
                                range={range}
                                DayComponent={DayComponent}
                                onClick={this.onClick}
                                showYear={
                                    i === 0 || month.year() !== today.year()
                                } // Первый месяц в маске или все месяцы других годов
                            />
                        </div>
                    ))}
                </div>
            </Modal>
        );
    }

    render() {
        return this.props.opened ? (
            <ModalHistoryWrapper onClose={this.props.onClose}>
                {this.getCalendar()}
            </ModalHistoryWrapper>
        ) : (
            this.getCalendar()
        );
    }
}
